import gc
import sys
import uuid
import copy
import time
import bisect
import logging
import binascii
import itertools as it
import functools as ft
import collections

import cython  # noqa
import gevent.queue

from sandbox import common

import sandbox.common.types.task as ctt

import sandbox.serviceq.types as qtypes
import sandbox.serviceq.journal as qjournal

# version of saved state, increases on incompatible changes
VERSION = 17

SEM_GROUP_SEP = "/"
LAST_QUOTA_REMNANTS_LIMIT = 1000
QUOTA_RATIO_SCALE = 1000
RECENTLY_EXECUTED_JOBS_TTL = 300  # in seconds
MAX_OPLOG_SIZE = 30000  # maximum number of operations in oplog without followers
FLOATING_WINDOW_TIME_INTERVAL = 12 * 60 * 60
EXTERNAL_SCALE = 1000  # values requested via RPC divided by this coefficient, to avoid integer overflow
FRACTIONAL_SCALE = 10  # coefficient for avoid fractional computing resources values

# special owner and appropriate priorities for running important tasks to bypass the owners rating
UNLIMITED_OWNER = -1
UNLIMITED_OWNER_MINIMUM_PRIORITY = int(ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.LOW))
UNLIMITED_OWNER_MAXIMUM_PRIORITY = int(ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.HIGH))

inf = float("inf")


def remaining_quota_item(quota, consumed_qp):
    return [-consumed_qp * QUOTA_RATIO_SCALE / quota if quota else -inf, quota - consumed_qp]


# noinspection PyUnresolvedReferences
if cython.compiled:
    # noinspection PyUnresolvedReferences
    from sandbox.serviceq.queues import HostQueue, MergeQueue
else:
    class HostQueue(list):
        def push(self, item):
            i = bisect.bisect(self, item)
            if i > 0 and self[i - 1] == item:
                self[i - 1] = item
            else:
                self.insert(i, item)

        def cleanup(self):
            self[:] = [entry for entry in self if entry]

        def __iter__(self):
            for item in list.__iter__(self):
                if item.task_ref:
                    yield item

    class MergeQueue(object):
        def __init__(self, queues):
            self.__merged = sorted(it.chain.from_iterable(queues))

        def __iter__(self):
            for item in self.__merged:
                if item.task_ref:
                    yield item


def add_task_to_hosts_queue(state, task_id, priority, score, hosts, owner, task_ref):
    total_items = 0
    cluster_queue = state.cluster_queue
    host_queue = state.host_queue
    hosts_cache = state.hosts_cache
    if score is None:
        clusters = collections.defaultdict(list)
        for host_score, host in hosts:
            clusters[host_score].append(host)
    else:
        cached_hosts = hosts_cache.get(hosts)
        if cached_hosts is None:
            if len(hosts_cache) > state.MAX_HOSTS_CACHE_SIZE:
                hosts_cache.clear()
            cache_key = hosts
            hosts = hosts_cache[cache_key] = state.hosts.indexes_from_bits(hosts)
        else:
            hosts = cached_hosts
        clusters = {score: hosts}
    owners = [owner]
    if priority >= UNLIMITED_OWNER_MINIMUM_PRIORITY:
        owners.append(UNLIMITED_OWNER)
    for score, cluster in clusters.items():
        cluster = tuple(cluster)
        cluster_hash = hash(cluster)
        for owner in owners:
            by_cluster = cluster_queue.setdefault(cluster_hash, {})
            by_owner = by_cluster.setdefault(cluster, {})
            by_prio = by_owner.setdefault(owner, {})
            queue = by_prio.get(priority)
            if queue is None:
                queue = by_prio[priority] = HostQueue()
                item = (cluster, queue)
                for host in cluster:
                    host_queue.setdefault(host, {}).setdefault(owner, {}).setdefault(priority, []).append(item)
            queue.push(qtypes.HostQueueItem(-score, task_id, task_ref))
            total_items += 1
    return total_items


def quota_downscale(qp, use_cores=False):
    return (
        qp * 1000 // FRACTIONAL_SCALE // FLOATING_WINDOW_TIME_INTERVAL
        if use_cores else
        qp // EXTERNAL_SCALE
    )


def quota_upscale(qp, use_cores=False):
    return (
        qp * FRACTIONAL_SCALE * FLOATING_WINDOW_TIME_INTERVAL // 1000  # qp == mcores
        if use_cores else
        qp * EXTERNAL_SCALE
    )


class LocalState(common.patterns.Abstract):
    __slots__ = (
        "last_quota_remnants",
        "locks",
        "prequeue",
        "complex_api_consumption",
        "locked_jobs_ids",
        "snapshot_id", "snapshot", "snapshot_watchdog",
    )
    __defs__ = (
        collections.defaultdict(ft.partial(collections.deque, maxlen=LAST_QUOTA_REMNANTS_LIMIT)),
        set(),
        qtypes.PreQueue(),
        qtypes.ComplexApiConsumption(),
        set(),
        0, None, None,
    )
    __copier__ = copy.deepcopy

    SNAPSHOT_WATCHDOG_TIMEOUT = 300  # in seconds

    def _snapshot_watchdog(self):
        gevent.sleep(self.SNAPSHOT_WATCHDOG_TIMEOUT)
        self.snapshot = None

    def start_snapshot_watchdog(self):
        self.stop_snapshot_watchdog()
        # noinspection PyAttributeOutsideInit
        self.snapshot_watchdog = gevent.spawn(self._snapshot_watchdog)

    def stop_snapshot_watchdog(self):
        if self.snapshot_watchdog is not None and not self.snapshot_watchdog.dead:
            self.snapshot_watchdog.kill()


# noinspection PyProtectedMember
class PersistentState(common.patterns.Abstract, qtypes.Serializable):
    __slots__ = (
        "hosts", "owners", "task_types", "client_tags",
        "task_queue", "task_queue_index",
        "host_queue", "cluster_queue", "hosts_cache",
        "host_capabilities", "task_res_sizes",
        "wants_semaphore", "semaphore_blockers",
        "unwanted_contenders",
        "consumptions", "owner_parents",
        "quotas", "owners_rating", "owners_rating_index", "owners_rating_reverse_index",
        "executions", "executing_tids", "recently_executed_jobs", "queue_size_by_owners",
        "semaphores", "semaphore_index", "auto_semaphores",
        "task_semaphores", "tasks_acquired",
        "semaphore_groups",
        "sequences", "db_operations",
        "api_quotas",
        "resources_cache",
        "web_api_quota",
        "resource_locks",
        "quota_pools",
        "operation_id", "common_oplog", "oplogs", "operations_checksum"
    )
    __defs__ = (
        qtypes.IndexedList(), qtypes.IndexedList(), qtypes.IndexedList(), qtypes.IndexedList(),
        {}, {},
        {}, {}, {},
        {}, {},
        {}, qtypes.SemaphoreBlockers(),
        [],
        {}, {},
        collections.defaultdict(dict), collections.defaultdict(list), {}, collections.defaultdict(dict),
        {}, set(), collections.OrderedDict(), collections.Counter(),
        {}, {}, {},
        {}, set(),
        {SEM_GROUP_SEP: qtypes.SemaphoreGroup(set(), set())},
        collections.Counter(), qtypes.DBOperations(),
        {},
        None,
        None,
        qtypes.ResourceLocks(),
        qtypes.QuotaPools(),
        0, collections.deque(), {}, 0
    )
    __copier__ = copy.deepcopy

    MAX_HOSTS_CACHE_SIZE = 10000

    Snapshot = collections.namedtuple("Snapshot", ("data", "operation_id"))
    Snapshot.__new__.__defaults__ = (None, 0)

    FollowerOplog = common.patterns.namedlist("FollowerOplog", ("oplog", "update_time"))

    Stats = collections.namedtuple(
        "Stats",
        (
            "tasks", "hosts",
            "clusters", "cluster_max_size", "cluster_p95_5_sizes",
            "task_queue_size", "task_queue_mem",
            "cluster_queue_size", "cluster_queue_mem",
            "resources", "common_oplog", "oplogs", "gc_counts"
        )
    )

    @property
    def stats(self):
        task_queue_item_size = sys.getsizeof(qtypes.TaskQueueItem._fields)
        task_queue_hosts_item_size = sys.getsizeof(qtypes.TaskQueueHostsItem._fields)
        task_queue_size = sum(it.imap(lambda _: len(_.hosts), it.chain.from_iterable(self.task_queue.itervalues())))
        task_queue_mem = sum((
            sys.getsizeof(self.task_queue),
            sum(sys.getsizeof(q) for q in self.task_queue.itervalues()),
            sys.getsizeof(self.task_queue_index),
            sum(len(q) for q in self.task_queue.itervalues()) * task_queue_item_size,
            task_queue_size * task_queue_hosts_item_size,
            sum(
                sys.getsizeof(_.task_info.requirements) + sys.getsizeof(
                    _.task_info.requirements and _.task_info.requirements.resources
                )
                for _ in it.chain.from_iterable(self.task_queue.itervalues())
            )
        ))

        host_queue_item_size = sys.getsizeof(qtypes.HostQueueItem)
        resources = len(set(
            it.chain.from_iterable(
                (_.task_info.requirements and _.task_info.requirements.resources) or ()
                for _ in it.chain.from_iterable(self.task_queue.itervalues())
            )
        ))

        clusters_sizes = []
        clusters_queues_total = 0
        clusters_queues_items_total = 0
        for _, by_cluster in self.cluster_queue.iteritems():
            for cluster, by_owner in by_cluster.iteritems():
                clusters_sizes.append(len(cluster))
                for by_prio in by_owner.itervalues():
                    clusters_queues_total += len(by_prio)
                    clusters_queues_items_total += sum(len(_) for _ in by_prio.itervalues())
        clusters_queues_mem = clusters_queues_items_total * host_queue_item_size
        clusters_sizes.sort()
        clusters_total = len(clusters_sizes)
        cluster_max_size = None
        cluster_p95_5_sizes = []
        if clusters_sizes:
            cluster_max_size = clusters_sizes[-1]
            for i in range(95, 0, -5):
                cluster_p95_5_sizes.append(clusters_sizes[clusters_total * i / 100])

        return self.Stats(
            len(self.task_queue_index), len(self.host_queue),
            clusters_total, cluster_max_size, cluster_p95_5_sizes,
            task_queue_size, common.utils.size2str(task_queue_mem),
            clusters_queues_items_total, common.utils.size2str(clusters_queues_mem),
            resources, len(self.common_oplog),
            {s: (len(o), int(time.time() - u)) for s, (o, u) in self.oplogs.iteritems()}, gc.get_count()
        )

    # noinspection PyDefaultArgument
    @staticmethod
    def __operation(method, operations={}):
        method_hash = binascii.crc32(method.__name__)
        assert method_hash not in operations, "Operation {} has the same hash {} as {}".format(
            method.__name__, method_hash, operations[method_hash].__name__
        )
        operations[method_hash] = method

        @ft.wraps(method)
        def wrapper(self, server, *args):
            self.operation_id += 1
            prev_operation_checksum, self.operations_checksum = self.operations_checksum, binascii.crc32(
                str(method_hash),
                binascii.crc32(
                    str(self.operation_id),
                    self.operations_checksum
                )
            )
            operation = qjournal.Operation(
                self.operation_id, method_hash,
                [(arg.encode() if isinstance(arg, qtypes.Serializable) else arg) for arg in args],
                prev_operation_checksum,
                self.operations_checksum
            )
            self.common_oplog.append(operation)
            min_operation_id = server.snapshot_operation_id
            while len(self.common_oplog) > MAX_OPLOG_SIZE and min_operation_id >= self.common_oplog[0].operation_id:
                self.common_oplog.popleft()
            for oplog, _ in self.oplogs.itervalues():
                oplog.append(operation)
            server._logger.info("New operation #%s [%s]", operation.operation_id, method.__name__)
            server.add_to_journal(operation)
            return method(self, server, *args)
        return wrapper
    # noinspection PyUnresolvedReferences
    __operation = __operation.__func__

    @property
    def operations(self):
        return self.__operation.__defaults__[0]

    # noinspection PyAttributeOutsideInit
    def apply(self, server, operation_id, method_hash, args, _, checksum):
        assert self.operation_id + 1 == operation_id, "Bad operation order {} -> {}".format(
            self.operation_id, operation_id
        )
        self.operation_id = operation_id
        self.operations_checksum = checksum
        self.operations[method_hash](self, server, *args)

    @__operation
    def add_task(self, _, task_id, priority, hosts, task_info, score):
        task_info = qtypes.TaskInfo.decode(task_info)
        for sem_id in task_info.semaphores or ():
            self.wants_semaphore.setdefault(sem_id, set()).add(task_id)
        self.__del_task(task_id)
        self.queue_size_by_owners[task_info.owner] += 1
        task_ref = qtypes.TaskRef(task_id)

        if task_info.requirements and task_info.requirements.resources:
            self.task_res_sizes[task_id] = sum(it.ifilter(None, task_info.requirements.resources.itervalues()))
            # noinspection PyAttributeOutsideInit
            self.resources_cache = None
        item = self.task_queue_index[task_id] = qtypes.TaskQueueItem(
            task_id, priority, hosts, task_ref, task_info, score
        )
        bisect.insort(self.task_queue.setdefault(priority, []), item)
        add_task_to_hosts_queue(
            self, task_id, priority, score, hosts, task_info.owner, task_ref
        )

    def __del_task(self, task_id):
        item = self.task_queue_index.pop(task_id, None)
        self.task_res_sizes.pop(task_id, None)
        if item is not None:
            self.queue_size_by_owners[item.task_info.owner] -= 1
            item.task_ref.clear()
        self.semaphore_blockers.remove(task_id)

    @__operation
    def del_task(self, _, task_id):
        self.__del_task(task_id)

    @__operation
    def add_host(self, _, host):
        return self.hosts.append(host)

    def add_host_(self, server, host, internal=False):
        index = self.hosts.index.get(host)
        if index is None:
            index = (
                self.hosts.append(host)
                if internal else
                self.add_host(server, host)
            )
        return index

    @__operation
    def add_hosts(self, _, hosts):
        for host in hosts:
            self.hosts.append(host)

    def add_hosts_(self, server, hosts):
        new_hosts = []
        for host in hosts:
            if host not in self.hosts:
                new_hosts.append(host)
        if new_hosts:
            self.add_hosts(server, new_hosts)

    @__operation
    def add_task_owner(self, _, owner):
        return self.owners.append(owner)

    def add_task_owner_(self, server, owner):
        index = self.owners.index.get(owner)
        if index is None:
            index = self.add_task_owner(server, owner)
        return index

    def __sort_owner_ratings(self):
        for owners_rating in self.owners_rating.values():
            owners_rating.sort()

    @__operation
    def set_parent_owner(self, _, owner_index, parent_index):
        prev_parent_index = self.owner_parents.pop(owner_index, None)
        if parent_index is None:
            if prev_parent_index is not None:
                for rating_item in self.owners_rating_reverse_index.get(owner_index, {}).values():
                    if rating_item and len(rating_item) > 1:
                        del rating_item[-2]
        else:
            self.owner_parents[owner_index] = parent_index
            for pool, rating_item in self.owners_rating_reverse_index.get(owner_index, {}).items():
                parent_quota = self.quotas.get(parent_index, {}).get(pool)
                if rating_item and parent_quota:
                    rating_item[:] = [parent_quota[1], rating_item[-1]]

        # Adjust parent quotas
        for pool, owner_quota in self.quotas.get(owner_index, {}).items():
            owner_quota = owner_quota[0]
            if prev_parent_index is not None:
                self.__add_quota(prev_parent_index, -owner_quota, pool)
            if parent_index is not None:
                self.__add_quota(parent_index, owner_quota, pool)

        self.__sort_owner_ratings()

    @__operation
    def sort_owners_rating(self, _):
        for owner, pool_quotas in self.quotas.iteritems():
            for pool, (quota, remaining_quota, _) in pool_quotas.iteritems():
                consumption = self.consumptions.get(owner, {}).get(pool)
                if consumption is not None:
                    consumed_qp = sum(consumption.qp)
                    remaining_quota[:] = remaining_quota_item(quota, consumed_qp)

        self.__sort_owner_ratings()

    @__operation
    def add_task_type(self, _, task_type):
        return self.task_types.append(task_type)

    def add_task_type_(self, server, task_type):
        index = self.task_types.index.get(task_type)
        if index is None:
            index = self.add_task_type(server, task_type)
        return index

    @__operation
    def add_client_tags(self, _, client_tags):
        return self.client_tags.append(client_tags)

    def add_client_tags_(self, server, client_tags):
        index = self.client_tags.index.get(client_tags)
        if index is None:
            index = self.add_client_tags(server, client_tags)
        return index

    @__operation
    def del_wanted_semaphores(self, _, task_id, sem_ids):
        self.semaphore_blockers.remove(task_id)
        for sem_id in sem_ids:
            self.wants_semaphore.get(sem_id, set()).discard(task_id)
            if not self.wants_semaphore.get(sem_id):
                self.wants_semaphore.pop(sem_id, None)

    @__operation
    def update_host_capabilities(self, _, host, capabilities):
        self.host_capabilities[host] = qtypes.ComputingResources.decode(capabilities)

    def update_host_capabilities_(self, server, host, capabilities):
        if capabilities and self.host_capabilities.get(host) != capabilities:
            self.update_host_capabilities(server, host, capabilities)

    def __cleanup_recently_executed_job(self):
        deadline = time.time() - RECENTLY_EXECUTED_JOBS_TTL
        for job_id, timestamp in self.recently_executed_jobs.iteritems():
            if timestamp > deadline:
                break
            self.recently_executed_jobs.pop(job_id)

    def __add_recently_executed_job(self, job_id):
        self.__cleanup_recently_executed_job()
        self.recently_executed_jobs[job_id] = time.time()

    def __job_recently_executed(self, job_id):
        self.__cleanup_recently_executed_job()
        return job_id in self.recently_executed_jobs

    def __start_consumption(
        self, owner, pool, task_id, start_time, job_id, qp, task_info, ram, cpu, hdd, ssd, add_execution=False
    ):
        consumption = self.consumptions.setdefault(owner, {}).setdefault(
            pool, qtypes.Consumption(FLOATING_WINDOW_TIME_INTERVAL)
        )
        started = consumption.started(start_time, job_id, qp, task_info.duration, ram, cpu, hdd, ssd)
        if started:
            parent_owner = self.owner_parents.get(owner)
            if parent_owner is not None:
                parent_consumption = self.consumptions.setdefault(parent_owner, {}).setdefault(
                    pool,
                    qtypes.Consumption(FLOATING_WINDOW_TIME_INTERVAL)
                )
                parent_consumption.started(start_time, job_id, qp, task_info.duration, ram, cpu, hdd, ssd)
            if add_execution:  # TODO: remove after SANDBOX-8910
                self.executions[job_id] = [owner, task_id, pool]
                self.executing_tids.add(task_id)
            return True
        return False

    @__operation
    def start_execution(self, server, start_time, job_id, task_id, qp, ram, cpu, hdd, ssd, pool=None):
        job_id_hex = job_id
        job_id = uuid.UUID(hex=job_id).bytes
        task_info = self.task_queue_index[task_id].task_info
        owner = task_info.owner
        if owner is None:
            server._logger.warning("Empty owner for task #%s (%s)", task_id, job_id_hex)
            return True
        if task_id in self.executing_tids:
            return None
        if job_id in self.executions or self.__job_recently_executed(job_id):
            return False
        result = None
        # TODO: remove pool None after switch to pool quotas
        for i, pool in enumerate([pool] if pool is None else [pool, None]):
            result = self.__start_consumption(
                owner, pool, task_id, start_time, job_id, qp, task_info, ram, cpu, hdd, ssd, add_execution=not i
            )
        return result

    def __finish_consumption(self, result, owner, pool, task_id, finish_time, job_id):
        consumption = self.consumptions.get(owner, {}).get(pool)
        if consumption:
            consumed = consumption.finished(finish_time, job_id)
            result.append(qtypes.FinishExecutionInfo(
                id=task_id,
                finished=finish_time,
                consumption=consumed.qp,
                ram=consumed.ram,
                cpu=consumed.cpu,
                hdd=consumed.hdd,
                ssd=consumed.ssd,
                pool=self.quota_pools.pools[pool] if pool is not None else None,
            ))
        parent_owner = self.owner_parents.get(owner)
        if parent_owner is not None:
            parent_consumption = self.consumptions.get(parent_owner, {}).get(pool)
            if parent_consumption:
                parent_consumption.finished(finish_time, job_id)

    @__operation
    def finish_execution(self, server, finish_time, job_id):
        result = []
        for job_id_hex in common.utils.chain(job_id):
            job_id = uuid.UUID(hex=job_id_hex).bytes
            execution_info = self.executions.pop(job_id, None)
            if execution_info is None:
                server._logger.warning("Execution info is not found while finishing job %s", job_id_hex)
                continue
            owner, task_id, pool = execution_info
            self.executing_tids.discard(task_id)
            self.__add_recently_executed_job(job_id)
            # TODO: remove pool None after switch to pool quotas
            for pool in [pool] if pool is None else [pool, None]:
                self.__finish_consumption(result, owner, pool, task_id, finish_time, job_id)
        return result

    @__operation
    def calculate_qp(self, _, now):
        consumptions = []
        for owner, pool_consumption in self.consumptions.iteritems():
            for pool, consumption in pool_consumption.iteritems():
                consumptions.append(((owner, pool), consumption.calculate(now)))
        return consumptions

    def __set_quota(self, owner, quota, pool, default):
        consumption = self.consumptions.get(owner, {}).get(pool)
        consumed_qp = sum(consumption.qp) if consumption else 0
        owner_quota = self.quotas.get(owner, {}).get(pool)
        if owner_quota:
            parent = self.owner_parents.get(owner)
            if parent is not None:
                self.__add_quota(parent, quota - owner_quota[0], pool)
            owner_quota[0], remaining_quota, owner_quota[2] = quota, owner_quota[1], default
            remaining_quota[:] = remaining_quota_item(quota, consumed_qp)
            self.__sort_owner_ratings()
        else:
            remaining_quota = remaining_quota_item(quota, consumed_qp)
            rating_item = [remaining_quota]
            self.add_to_owners_rating(owner, pool, quota, rating_item, default)

    def __add_quota(self, owner, quota, pool):
        prev_quota = self.quotas[owner][pool][0]
        self.__set_quota(owner, prev_quota + quota, pool, False)

    @__operation
    def set_quota(self, _, owner, quota, pool=None, default=False):
        self.__set_quota(owner, quota, pool, default)

    @__operation
    def add_quota_pool(self, _, pool, tags):
        self.quota_pools.add(pool, tags)

    @__operation
    def update_quota_pool(self, _, pool, tags, default=None):
        self.quota_pools.update(pool, tags, default)

    @__operation
    def add_semaphore_blockers(self, _, blockers):
        self.semaphore_blockers.add(blockers)

    @staticmethod
    def __semaphore_group(name):
        return name.rpartition(SEM_GROUP_SEP)[0] or SEM_GROUP_SEP

    @classmethod
    def __add_semaphore_to_group(cls, sem_id, name, semaphore_groups):
        group = cls.__semaphore_group(name)
        if group == SEM_GROUP_SEP:
            semaphore_groups[group].sem_ids.add(sem_id)
            return
        parent = None
        sem_group = None
        for item in group.split(SEM_GROUP_SEP):
            gr = SEM_GROUP_SEP.join((parent, item)) if parent else item
            sem_group = semaphore_groups.setdefault(gr, qtypes.SemaphoreGroup(set(), set()))
            semaphore_groups[parent or SEM_GROUP_SEP].groups.add(gr)
            parent = gr
        sem_group.sem_ids.add(sem_id)

    @classmethod
    def __remove_semaphore_from_group(cls, sem_id, name, semaphore_groups):
        group = cls.__semaphore_group(name)
        if group == SEM_GROUP_SEP:
            semaphore_groups[group].sem_ids.discard(sem_id)
            return
        semaphore_groups[group].sem_ids.discard(sem_id)
        items = group.split(SEM_GROUP_SEP)
        sem_group = None
        group_name = None
        for i in range(len(items), -1, -1):
            subgroup = sem_group
            subgroup_name = group_name
            group_name = SEM_GROUP_SEP.join(items[:i]) or SEM_GROUP_SEP
            sem_group = semaphore_groups[group_name]
            if subgroup:
                sem_group.groups.discard(subgroup_name)
            if sem_group.sem_ids or sem_group.groups:
                break

    @__operation
    def add_semaphore(self, _, sem_id, name, owner, capacity, auto, shared, public):
        if sem_id in self.semaphores:
            return False
        sem = self.semaphores[sem_id] = qtypes.Semaphore((name, owner, capacity, 0, auto, shared, public, {}))
        self.semaphore_index[name] = sem_id
        if auto:
            self.auto_semaphores[sem_id] = sem
        self.__add_semaphore_to_group(sem_id, name, self.semaphore_groups)
        return True

    @__operation
    def set_api_quota(self, _, login, api_quota):
        if api_quota is None:
            self.api_quotas.pop(login, None)
        else:
            self.api_quotas[login] = api_quota

    @__operation
    def set_web_api_quota(self, _, api_quota):
        # noinspection PyAttributeOutsideInit
        self.web_api_quota = api_quota

    @__operation
    def acquire_resource_lock(self, _, resource_id, host, timestamp):
        return self.resource_locks.acquire(resource_id, host, timestamp)

    @__operation
    def release_resource_lock(self, _, resource_id, host):
        return self.resource_locks.release(resource_id, host)

    @__operation
    def clean_resource_locks(self, _, timestamp):
        self.resource_locks.clean_resource_locks(timestamp)

    @__operation
    def remove_semaphore(self, _, sem_id):
        self.auto_semaphores.pop(sem_id, None)
        sem = self.semaphores.pop(sem_id, None)
        if not sem:
            return False
        self.semaphore_index.pop(sem.name, None)
        self.__remove_semaphore_from_group(sem_id, sem.name, self.semaphore_groups)
        return True

    @__operation
    def update_semaphore(self, _, sem_id, owner, capacity, auto, shared, public):
        sem = self.semaphores.get(sem_id)
        if not sem:
            return
        if sem.auto != auto:
            if auto:
                self.auto_semaphores[sem_id] = sem
            else:
                self.auto_semaphores.pop(sem_id, None)
        if owner is not None:
            sem.owner = owner
        sem.auto = auto
        sem.capacity = capacity
        if shared is not None:
            sem.shared = shared
        if public is not None:
            sem.public = public

    def check_semaphores_capacity(self, task_id, only_check=True, logger=None):
        """
        :return: (<can acquire semaphores>, <blocked sem_id>, <required weight>)
        """
        try:
            semaphores = self.task_semaphores[task_id]
        except KeyError:
            if logger:
                logger.exception("Unexpected error")
            return False, None, None, None
        rollback = []
        for acquire in semaphores.acquires:
            # acquire.name in fact contains semaphore id
            sem_id = acquire.name
            try:
                sem = self.semaphores[sem_id]
            except KeyError:
                if logger:
                    logger.exception("Unexpected error (task #%s: %r)", task_id, semaphores)
                break
            if task_id in sem.tasks:
                break
            if sem.value + acquire.weight > sem.capacity:
                break
            rollback.append((sem_id, sem, acquire.weight))
            if not only_check:
                sem.tasks[task_id] = acquire.weight
                sem.value += acquire.weight
        else:
            if not only_check:
                self.tasks_acquired.add(task_id)
            return rollback, None, None, None
        if not only_check:
            # rollback changes
            for _, sem, weight in rollback:
                sem.value -= weight
                sem.tasks.pop(task_id)
        return False, sem_id, acquire.weight, semaphores

    @__operation
    def acquire_semaphores(self, server, task_id):
        return self.check_semaphores_capacity(task_id, only_check=False, logger=server._logger)

    @__operation
    def release_semaphores(self, server, task_id):
        try:
            semaphores = self.task_semaphores[task_id]
        except KeyError:
            server._logger.exception("Unexpected error")
            return False
        rollback = []
        sem_ids = []
        for acquire in semaphores.acquires:
            # acquire.name actually contains semaphore id
            sem_id = acquire.name
            sem_ids.append(sem_id)
            sem = self.semaphores[sem_id]
            weight = sem.tasks.pop(task_id, None)
            if weight is None:
                continue
            rollback.append((sem, weight))
            sem.value -= weight
        if rollback:
            self.tasks_acquired.discard(task_id)
            self.task_semaphores.pop(task_id, None)
            return rollback, sem_ids
        # rollback changes
        for sem, weight in rollback:
            sem.value += weight
            sem.tasks[task_id] = weight
        return False

    @__operation
    def set_task_semaphores(self, _, task_id, semaphores):
        if semaphores is None:
            self.task_semaphores.pop(task_id, None)
        else:
            self.task_semaphores[task_id] = qtypes.TaskSemaphores.decode(semaphores)

    @__operation
    def next_id(self, _, name):
        self.sequences[name] += 1
        return self.sequences[name]

    def push_db_operation_(self, server, operation, logger=None):
        if logger is None:
            logger = server._logger
        logger.info("New DB operation: %s", operation)
        self.push_db_operation(server, operation)

    @__operation
    def push_db_operation(self, _, operation):
        self.db_operations.push(qtypes.DBOperations.Operation.decode(operation))

    @__operation
    def pop_db_operation(self, server, queue_name):
        queue = self.db_operations[queue_name]
        try:
            queue.get_nowait()
        except gevent.queue.Empty:
            server._logger.critical(
                "Impossible error: trying to get items from empty DB operations queue %s", queue_name
            )

    @__operation
    def set_unwanted_contenders(self, _, unwanted_contenders):
        self.unwanted_contenders[:] = unwanted_contenders if unwanted_contenders else []

    @__operation
    def flush_billing_metrics(self, _, owner, now):
        return {
            pool: consumption.calculate_usage_metrics(now, flush=True)
            for pool, consumption in self.consumptions.get(owner, {}).iteritems()
        }

    def collect_billing_metrics(self, server, owner_to_abc_id, switcher, config):
        billing_metrics = []
        parents = set(self.owner_parents.values())
        for owner_index, quotas_by_pool in self.quotas.iteritems():
            usage_metrics_by_pool = self.flush_billing_metrics(server, owner_index, int(time.time()))
            if owner_index in parents:
                continue
            owner = self.owners[owner_index]
            for pool_index, quota_item in quotas_by_pool.iteritems():
                if pool_index is None:
                    continue
                quota, _, default = quota_item
                if default:
                    continue
                abc_id = owner_to_abc_id[owner]
                if abc_id is None:
                    continue
                pool = self.quota_pools.pools[pool_index]
                segment = config.serviceq.server.quotas.billing.pool_to_segment[pool]
                metric_base = dict(
                    abc_id=str(abc_id),
                    schema="sandbox.slot.v1",
                    source_wt=int(time.time()),
                    source_id=config.this.fqdn,
                    version="v1alpha1",
                    labels=dict(account=owner)
                )
                for usage_metric in map(qtypes.UsageMetric, usage_metrics_by_pool.get(pool_index, ())):
                    interval = usage_metric.finish_time - usage_metric.start_time
                    billing_metrics.append(dict(
                        metric_base,
                        id=str(uuid.uuid1()),
                        tags=dict(
                            segment=segment,
                            metric_type="guarantee",
                        ),
                        usage=dict(
                            quantity=str(int(quota_downscale(quota * interval, use_cores=True) // 1000)),
                            unit="vcpu*second",
                            start=usage_metric.start_time,
                            finish=usage_metric.finish_time,
                            type="delta",
                        ),
                    ))
                    billing_metrics.append(dict(
                        metric_base,
                        id=str(uuid.uuid1()),
                        tags=dict(
                            segment=segment,
                            metric_type="usage",
                        ),
                        usage=dict(
                            quantity=str(int(usage_metric.usage // FRACTIONAL_SCALE)),
                            unit="vcpu*second",
                            start=usage_metric.start_time,
                            finish=usage_metric.finish_time,
                            type="delta",
                        ),
                    ))
            switcher.switch()
        return billing_metrics

    def add_to_owners_rating(self, owner, pool, quota, rating_item, default, insort=True):
        self.quotas[owner][pool] = [quota, rating_item[-1], default]
        self.owners_rating_index[id(rating_item)] = (owner, pool)
        self.owners_rating_reverse_index[owner][pool] = rating_item
        if insort:
            bisect.insort(self.owners_rating[pool], rating_item)
        else:
            self.owners_rating[pool].append(rating_item)

    # noinspection PyAttributeOutsideInit
    def reset_quotas(self):
        self.quotas = collections.defaultdict(dict)
        self.owners_rating = collections.defaultdict(list)
        self.owners_rating_index = {}
        self.owners_rating_reverse_index = collections.defaultdict(dict)

    @property
    def resources(self):
        if self.resources_cache is None:
            # noinspection PyAttributeOutsideInit
            self.resources_cache = list(set(
                it.chain.from_iterable(
                    (_.task_info.requirements and _.task_info.requirements.resources) or ()
                    for _ in it.chain.from_iterable(self.task_queue.itervalues())
                )
            ))
        return self.resources_cache

    def encode(self):
        cluster_index = {}
        cluster_queue = []
        clusters = []
        for cluster_hash, by_hash in self.cluster_queue.iteritems():
            for cluster in by_hash.iterkeys():
                clusters.append((cluster, cluster_hash))
        for cluster, cluster_hash in sorted(clusters):
            queues = self.cluster_queue[cluster_hash][cluster]
            cluster_index[cluster] = len(cluster_index)
            cluster_queue.append([
                cluster,
                {
                    owner: {
                        prio: [_.encode() for _ in queue if _]
                        for prio, queue in by_owner.iteritems()
                    }
                    for owner, by_owner in queues.iteritems()
                }
            ])
        return (
            self.operation_id,
            self.task_queue,
            {
                host: {
                    owner: {
                        prio: [cluster_index[cluster] for cluster, _ in queues]
                        for prio, queues in by_prio.iteritems()
                    }
                    for owner, by_prio in by_owner.iteritems()
                }
                for host, by_owner in self.host_queue.iteritems()
            },
            self.hosts,
            self.host_capabilities,
            self.task_res_sizes,
            {k: tuple(v) for k, v in self.wants_semaphore.iteritems()},
            self.unwanted_contenders,
            {
                owner: {
                    pool: consumption.encode()
                    for pool, consumption in pool_consumption.iteritems()
                }
                for owner, pool_consumption in self.consumptions.iteritems()
            },
            self.owners,
            self.task_types,
            {
                owner: {pool: (None if default else quota) for pool, (quota, _, default) in pool_quotas.iteritems()}
                for owner, pool_quotas in self.quotas.iteritems()
            },
            self.executions,
            self.owner_parents,
            cluster_queue,
            self.client_tags,
            {sem_id: sem.encode() for sem_id, sem in self.semaphores.iteritems()},
            self.task_semaphores,
            list(self.tasks_acquired),
            dict(self.sequences),
            self.db_operations.encode(),
            self.api_quotas,
            self.operations_checksum,
            self.web_api_quota,
            self.resource_locks.encode(),
            self.quota_pools.encode(),
        )

    @classmethod
    def decode(cls, data, logger=None):
        logger = logger.getChild("state.decode") if logger else logging
        (
            operation_id,
            task_queue_,
            host_queue,
            hosts,
            host_capabilities,
            task_res_sizes,
            wants_semaphore,
            unwanted_contenders,
            consumptions,
            owners,
            task_types,
            quotas_,
            executions,
            owner_parents,
            cluster_queue_,
            client_tags,
            semaphores,
            task_semaphores,
            tasks_acquired,
            sequences,
            db_operations,
            api_quotas,
            operations_checksum,
            web_api_quota,
            resource_locks,
            quota_pools,
        ) = data
        task_queue_index = {}
        task_queue = {}
        task_queue_size = 0
        task_refs = {}
        for priority, queue in task_queue_.iteritems():
            pqueue = []
            for item in queue:
                if not item[3]:
                    continue
                task_id = item[0]
                item[3] = task_refs[task_id] = qtypes.TaskRef(task_id)
                # TODO: remove after SANDBOX-9402
                if isinstance(item[2], list) and item[5] is not None:
                    item[2] = qtypes.make_bits_from_indexes(item[2])
                item = qtypes.TaskQueueItem.decode(item)
                pqueue.append(item)
                task_queue_index[item.task_id] = item
            task_queue[priority] = pqueue
            task_queue_size += len(pqueue)
        logger.info("task_queue: %d item(s)", task_queue_size)

        host_queue_total_items = 0
        cluster_index = []
        cluster_queue = {}
        for cluster, by_owners in cluster_queue_:
            cluster = tuple(cluster)
            cluster_queue.setdefault(hash(cluster), {})[cluster] = by_owners
            cluster_index.append((cluster, hash(cluster)))
            for owner, by_prio in by_owners.iteritems():
                for prio, queue in by_prio.items():
                    queue_ = HostQueue()
                    for item in queue:
                        task_id = item[1]
                        task_ref = task_refs.get(task_id)
                        if task_ref is None:
                            continue
                        item[2] = task_ref
                        queue_.push(qtypes.HostQueueItem(*item))
                    by_prio[prio] = queue_
        for host, by_owner in host_queue.iteritems():
            for owner, by_prio in by_owner.iteritems():
                for prio, queues in by_prio.iteritems():
                    for i, index in enumerate(queues):
                        cluster, cluster_hash = cluster_index[index]
                        by_cluster = cluster_queue[cluster_hash]
                        if len(by_cluster) == 1:
                            by_owner = by_cluster.values()[0]
                        else:
                            by_owner = by_cluster[cluster]
                        queues[i] = (cluster, by_owner[owner][prio])
                    host_queue_total_items += sum(len(queue) for _, queue in queues)
        logger.info("host_queue: %d item(s)", host_queue_total_items)

        for owner, consumption in consumptions.items():
            for pool, pool_consumption in consumption.iteritems():
                consumptions[owner][pool] = qtypes.Consumption.decode(pool_consumption)
        logger.info("consumptions: %d item(s)", sum(len(c) for c in consumptions.itervalues()))

        # process quotas
        quota_pools = qtypes.QuotaPools.decode(quota_pools)
        owners_rating = collections.defaultdict(list)  # key - pool, value - rating
        owners_rating_index = {}  # key - rating item unique id, value - (owner, pool)
        owners_rating_reverse_index = collections.defaultdict(dict)  # key - owner, value - {pool: rating_item}
        quotas = collections.defaultdict(dict)
        for owner, quota in quotas_.iteritems():
            for pool, pool_quota in quota.iteritems():
                default = pool_quota is None
                if default:
                    pool_quota = quota_pools.default(pool)
                # TODO: remove after SANDBOX-8907
                elif pool is None and pool_quota == qtypes.DEFAULT_QUOTA:
                    default = True
                consumption = consumptions.get(owner, {}).get(pool)
                consumed_qp = (sum(consumption.qp) if consumption else 0)
                remaining_quota = remaining_quota_item(pool_quota, consumed_qp)
                rating_item = [remaining_quota]
                quotas[owner][pool] = [pool_quota, remaining_quota, default]
                owners_rating[pool].append(rating_item)
                owners_rating_index[id(rating_item)] = (owner, pool)
                owners_rating_reverse_index[owner][pool] = rating_item

        for owner, parent in owner_parents.iteritems():
            for pool, rating_item in owners_rating_reverse_index[owner].items():
                parent_quota = quotas.get(parent, {}).get(pool)
                if rating_item and parent_quota:
                    rating_item[:] = [parent_quota[1], rating_item[-1]]
        for pool_rating in owners_rating.itervalues():
            pool_rating.sort()

        queue_size_by_owners = collections.Counter()
        for item in task_queue_index.itervalues():
            queue_size_by_owners[item.task_info.owner] += 1

        semaphore_index = {}
        auto_semaphores = {}
        semaphore_groups = copy.deepcopy(cls.__defs__[cls.__slots__.index("semaphore_groups")])
        for sem_id, data in semaphores.viewitems():
            sem = semaphores[sem_id] = qtypes.Semaphore.decode(data)
            semaphore_index[sem.name] = sem_id
            cls.__add_semaphore_to_group(sem_id, sem.name, semaphore_groups)
            if sem.auto:
                auto_semaphores[sem_id] = sem

        for task_id, data in task_semaphores.viewitems():
            task_semaphores[task_id] = qtypes.TaskSemaphores.decode(data)

        sequences = collections.Counter(sequences)
        logger.info("semaphores: %d item(s)", len(semaphores))
        logger.info("auto_semaphores: %d item(s)", len(auto_semaphores))
        logger.info("task_semaphores: %d item(s)", len(task_semaphores))
        logger.info("db_operations: %d item(s)", len(db_operations))
        db_operations = qtypes.DBOperations.decode(db_operations)

        return cls(
            operation_id=operation_id,
            task_queue=task_queue,
            task_queue_index=task_queue_index,
            host_queue=host_queue,
            hosts=qtypes.IndexedList(hosts),
            owners=qtypes.IndexedList(owners),
            task_types=qtypes.IndexedList(task_types),
            client_tags=qtypes.IndexedList(client_tags),
            host_capabilities={k: qtypes.ComputingResources.decode(v) for k, v in host_capabilities.iteritems()},
            task_res_sizes=task_res_sizes,
            wants_semaphore={k: set(v) for k, v in wants_semaphore.iteritems()},
            consumptions=consumptions,
            owner_parents=owner_parents,
            quotas=quotas,
            owners_rating=owners_rating,
            owners_rating_index=owners_rating_index,
            owners_rating_reverse_index=owners_rating_reverse_index,
            executions=executions,
            executing_tids={task_id for owner, task_id, pool in executions.itervalues()},
            queue_size_by_owners=queue_size_by_owners,
            cluster_queue=cluster_queue,
            semaphores=semaphores,
            semaphore_index=semaphore_index,
            auto_semaphores=auto_semaphores,
            semaphore_groups=semaphore_groups,
            task_semaphores=task_semaphores,
            tasks_acquired=set(tasks_acquired),
            sequences=sequences,
            db_operations=db_operations,
            api_quotas=api_quotas,
            operations_checksum=operations_checksum,
            web_api_quota=web_api_quota,
            resource_locks=qtypes.ResourceLocks(resource_locks),
            unwanted_contenders=unwanted_contenders or [],
            quota_pools=quota_pools,
        )

    @classmethod
    def from_snapshot(cls, path):
        with open(path, "rb") as f:
            raw_data = f.read()
        unpacker = common.data.msgpack_unpacker()
        unpacker.feed(raw_data)
        _, data = list(iter(unpacker))
        return cls.decode(data)
