from __future__ import print_function

import argparse

import setproctitle

from sandbox.services import modules
from sandbox.services.base import loader


def list_services(args):
    print("Available services:")
    for service_class in modules.service_registry.values():
        prefix = "* {}: ".format(service_class.name)
        print(prefix + service_class.description.replace("\n", "\n" + " " * len(prefix)))


def run_service(args):
    service_class = modules.service_registry[args.name]
    # Set process title so everyone knows which service is running
    setproctitle.setproctitle("[sandbox] Service {}".format(args.name))
    # Create service instance and run it
    service_instance = loader.load_service(service_class, args.stderr_log)
    if args.once:
        service_instance.load_service_state()
        service_instance.tick()
        service_instance.on_stop()
    else:
        service_instance.run_forever()


if __name__ == "__main__":
    parser = argparse.ArgumentParser()

    action = parser.add_subparsers(dest="action")

    parser_list = action.add_parser("list", help="List available services")
    parser_list.set_defaults(func=list_services)

    parser_run = action.add_parser("run", help="Run some service")
    parser_run.add_argument("--name", help="Service to run", choices=modules.service_registry.keys(), required=True)
    parser_run.add_argument("--stderr-log", help="Write logs to stderr", action="store_true")
    parser_run.add_argument("--once", help="Single run of service's tick", action="store_true")
    parser_run.set_defaults(func=run_service)

    args = parser.parse_args()
    args.func(args)
