import os
import signal
import threading

import logging
import logging.handlers

import flask

import sandbox.common.log as cl
import sandbox.common.types.misc as ctm

from sandbox import common
from sandbox.common import config, rest
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


__all__ = ("load_service",)


logger = logging.getLogger(__name__)


def setup_logging(service_name, stderr_log=False):
    settings = config.Registry()

    if stderr_log:
        handler = logging.StreamHandler()
    else:
        log_dir = os.path.join(settings.server.services.log.root, "core")
        log_path = os.path.join(log_dir, service_name + ".log")
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)

        if settings.server.services.log.rotate:
            handler = cl.GZipTimedRotatingFileHandler(log_path, when="midnight", backupCount=14)
        else:
            handler = logging.handlers.WatchedFileHandler(log_path)

    handler.setFormatter(logging.Formatter("%(asctime)s %(levelname)-7s (%(filename)s:%(lineno)s) %(message)s"))

    logging.root.handlers = []  # Somebody could have already initialized logging
    logging.root.addHandler(handler)
    logging.root.setLevel(logging.getLevelName(settings.server.services.log.level))

    exception_handler = cl.ExceptionSignalsSenderHandler()
    logging.root.addHandler(exception_handler)

    # Mute some verbose fellas
    logging.getLogger("urllib3").setLevel(logging.WARNING)
    logging.getLogger("kazoo.client").setLevel(logging.INFO)


def setup_database():
    settings = config.Registry()

    mapping.ensure_connection(
        uri=settings.server.mongodb.connection_url,
        max_pool_size=100,
    )


def setup_yasandbox():
    controller.initialize()


def setup_signal_handlers(service_instance):
    def handler(signum, frame):
        logger.info("Got SIGINT, stopping %s...", service_instance.name)
        service_instance.request_stop()

    signal.signal(signal.SIGINT, handler)
    signal.signal(signal.SIGUSR2, lambda *_: common.threading.dump_threads(logger))


def setup_api(service_instance):
    settings = config.Registry()
    port = settings.server.services.api.port

    if port is None:
        logger.info("Service API is disabled")
        return

    def status_view():
        return flask.json.jsonify(service_instance.get_status_report())

    app = flask.Flask(__name__)
    app.add_url_rule("/status", view_func=status_view, methods=[ctm.RequestMethod.GET])

    thread = threading.Thread(target=lambda: app.run(port=port))
    thread.daemon = True
    thread.start()


def load_service(service_class, stderr_log=False):
    # Check if path to local settings is defined
    config.ensure_local_settings_defined()
    # Now we're sure service exists, perform some initialization (logging, database, signals, api)
    setup_logging(service_class.name, stderr_log)
    setup_database()
    setup_yasandbox()

    # Set default component for all API requests
    rest.Client._default_component = ctm.Component.SERVICE

    service_instance = service_class()
    setup_signal_handlers(service_instance)
    setup_api(service_instance)

    return service_instance
