import os
import math
import logging
import threading

import kazoo.client
from kazoo.exceptions import LockTimeout


logger = logging.getLogger(__name__)


__all__ = ("FairZookeeperLock", "NoLock", "LockTimeout")


def noop():
    pass


class FairZookeeperLock(object):
    def __init__(self, path, name, owner, zk_hosts, on_disconnect=None, on_stepdown=None, watcher_delay=None):
        self.path = path
        self.name = name
        self.owner = owner
        self.on_disconnect = on_disconnect or noop
        self.on_stepdown = on_stepdown or noop
        self.watcher_delay = watcher_delay

        self.lock_path = os.path.join(path, name)

        self._client = self._create_zookeeper_client(zk_hosts)
        self._lock = self._client.Lock(self.lock_path, owner)

        self._watcher_thread = None
        self._watcher_event = threading.Event()

    def _create_zookeeper_client(self, hosts):
        def listener(state):
            if state != kazoo.client.KazooState.CONNECTED:
                self.on_disconnect()

        retry_policy = {"max_tries": 10, "max_delay": 10}
        client = kazoo.client.KazooClient(hosts=hosts, connection_retry=retry_policy)
        client.add_listener(listener)
        client.start(timeout=30)

        return client

    def _acquired_too_many(self):
        locks_acquired = 0
        total_locks = 0
        total_contenders = set()

        for name in self._client.get_children(self.path):
            path = os.path.join(self.path, name)
            contenders = self._client.Lock(path).contenders()

            if not contenders:
                continue  # Empty node, probably it's a stale lock

            if contenders[0] == self.owner:
                locks_acquired += 1

            total_locks += 1
            total_contenders |= set(contenders)

        logger.debug(
            "Acquired locks: %s out of %s, contenders: %s",
            locks_acquired, total_locks, len(total_contenders)
        )
        return locks_acquired > math.ceil(float(total_locks) / len(total_contenders))

    def _run_fair_watchdoge(self):
        def thread_func():
            # Delay check for some time to ensure service works at least for `watcher_delay` seconds
            if self.watcher_delay:
                logger.debug("Wait for %ss before load-balancing", self.watcher_delay)
                self._watcher_event.wait(self.watcher_delay)

            while not self._watcher_event.is_set():
                if self._acquired_too_many():
                    self.on_stepdown()
                # Perform check every 5 minutes
                self._watcher_event.wait(300)

        self._watcher_thread = threading.Thread(target=thread_func)
        self._watcher_thread.start()

    def acquire(self, timeout=None):
        logger.debug("Trying to lock '%s' (for '%s'), timeout %ss", self.lock_path, self.owner, timeout)
        self._lock.acquire(timeout=timeout)
        self._run_fair_watchdoge()
        logger.debug("Got lock, warming up...")

    def release(self):
        if self._client.state == kazoo.client.KazooState.CONNECTED:
            self._lock.release()
        else:
            logger.warning("Don't release lock because connection to Zookeeper is lost")

        self._watcher_event.set()
        self._watcher_thread.join()

    def destroy(self):
        self.release()
        self._client.stop()
        self._client.close()


class NoLock(object):
    def acquire(self, timeout=None):
        logger.info("Lock disabled, run like regular service")

    def release(self):
        pass

    def destroy(self):
        pass
