import abc
import time
import random
import logging
import calendar
import collections
import datetime as dt
import itertools as it

import six
# noinspection PyUnresolvedReferences,PyPackageRequirements
import setproctitle
import pymongo.errors

from sandbox.common import fs as common_fs
from sandbox.common import os as common_os
from sandbox.common import log as common_log
from sandbox.common import mds as common_mds
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import format as common_format
from sandbox.common import context as common_context
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import statistics as cts

from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.database import mapping

from sandbox.services import base

logger = logging.getLogger(__name__)


ResourceEntry = collections.namedtuple("ResourceEntry", ("id", "size"))
SleepLimits = collections.namedtuple("SleepLimits", ("max", "min"))


class BackupType(common_enum.Enum):
    BACKUP = None
    REPLICATION = None


class ResourceChunk(
    collections.namedtuple("Chunk", ("no", "parameters", "size", "resources", "tags"))
):
    """
    Note: size is in kilobytes
    """
    CHUNK_DESCRIPTION_TEMPLATE = "#{no} chunk of {resources_count} resources totally for {size}"

    @property
    def description(self):
        return self.CHUNK_DESCRIPTION_TEMPLATE.format(
            no=self.no, resources_count=self.amount, size=common_format.size2str(self.size << 10)
        )

    @property
    def amount(self):
        return len(self.resources)


class BaseBackupService(base.SingletonService):
    """ Base abstract for resources backup services. """
    __metaclass__ = abc.ABCMeta

    # Fine backup task statues
    OK_STATUSES = set(common_itertools.chain(
        ctt.Status.Group.QUEUE,
        ctt.Status.Group.EXECUTE,
        ctt.Status.Group.FINISH,
    ))

    BAD_TASK_TTL = 3  # (hours) Tasks in bad (non-fine) state time-to-live
    REST_API_TIMEOUT = 30  # (sec) Timeout for Sandbox REST requests
    MAX_BACKUP_CHUNK_SIZE = 300  # (GiB) Maximum chunk size for resource backup task

    class Counter(common_patterns.Abstract):
        """ Simple counters for statistics. """

        class Tasks(common_patterns.Abstract):
            """ Simple counters for tasks. """
            __slots__ = ["checked", "created"]
            __defs__ = [0] * 2

        class Resources(common_patterns.Abstract):
            """ Simple counters for resources. """
            __slots__ = ["checked", "operated", "size"]
            __defs__ = [0] * 3

        __slots__ = ["started", "tasks", "resources"]
        __defs__ = [0, None, None]

        def __init__(self):
            super(BaseBackupService.Counter, self).__init__()
            self.started = time.time()
            self.tasks = self.Tasks()
            self.resources = self.Resources()

        def on_task_create(self, chunk):
            self.tasks.created += 1
            self.resources.size += chunk.size
            self.resources.operated += chunk.amount

    def __init__(self, *args, **kwargs):
        super(BaseBackupService, self).__init__(*args, **kwargs)
        oauth_token = (
            common_fs.read_settings_value_from_file(self.sandbox_config.server.auth.oauth.token)
            if self.sandbox_config.server.auth.enabled else None
        )
        self.rest = common_rest.ThreadLocalCachableClient(
            auth=oauth_token, component=ctm.Component.SERVICE, total_wait=self.REST_API_TIMEOUT
        )

    @abc.abstractproperty
    def sleep_limits(self):  # type: () -> SleepLimits
        """ Sleep interval limits in seconds. """
        pass

    @abc.abstractproperty
    def task_priority(self):  # type: () -> ctt.Priority
        """ Priority for backup tasks, which will be created by this service. """
        pass

    @abc.abstractproperty
    def resource_processed_threshold(self):  # type: () -> int
        """ Amount of checked resources to select minimal sleep period. """
        pass

    @classmethod
    def _chunker(cls, chunk_logger, resources, parameters_base, tags):
        chunk_logger.debug("Splitting resources into chunks")
        chunk_counter = 0
        while resources:
            chunk_counter += 1
            chunk_size, resource_counter = 0, 0
            # Resource size in KiB, limit in GiB
            while chunk_size < cls.MAX_BACKUP_CHUNK_SIZE << 20 and resource_counter < len(resources):
                chunk_size += resources[resource_counter].size
                resource_counter += 1
            chunk_resource_ids = [
                r.id for r in sorted(resources[:resource_counter], key=lambda r: r.size, reverse=True)
            ]
            resources = resources[resource_counter:]
            yield ResourceChunk(chunk_counter, parameters_base, chunk_size, chunk_resource_ids, tags)
        if chunk_counter == 0:
            chunk_logger.info("Nothing to backup")
        else:
            chunk_logger.info("%d chunks prepared", chunk_counter)

    def _creator(self, chunks):
        if not chunks:
            return
        logger.debug("Creating backup tasks for %d chunks", len(chunks))
        total_size, total_resources = 0, 0
        max_size = max(max(chunk.size for chunk in chunks), 1)
        for chunk in chunks:
            parameters = chunk.parameters.copy()
            parameters.update(
                resource_id=",".join(map(str, chunk.resources)),
                chunk_size=chunk.size * 100 / max_size
            )
            task_id = self.rest.task(dict(
                type="BACKUP_RESOURCE_2",
                author=user_controller.User.service_user,
                owner=user_controller.Group.service_group,
                description=chunk.description,
                hidden=True,
                notifications=[],
                # REST API expects bytes here; add a 10MiB overhead just to be sure
                requirements={"disk_space": (chunk.size + (10 << 10)) << 10},
                custom_fields=[
                    {"name": key, "value": value}
                    for key, value in parameters.items()
                ],
                priority=self.task_priority,
                fail_on_any_error=True,
                tags=chunk.tags
            ))["id"]
            total_size += chunk.size
            total_resources += chunk.amount
            logger.info(
                "Task #%s '%s' created (subtotal is %s)",
                task_id, chunk.description, common_format.size2str(total_size << 10)
            )
            mapping.Resource.objects(
                id__in=chunk.resources,
                attributes__key="backup_task"
            ).update(
                set__attributes__S__value=task_id,
                set__time__updated=dt.datetime.utcnow()
            )
            self.rest.batch.tasks.start.update([task_id])
            yield task_id, chunk
        logger.info(
            "Created %d tasks totally for %s to backup %d resources",
            len(chunks), common_format.size2str(total_size << 10), total_resources
        )

    def _calculate_wait(self, counter):
        # Set next sleep step proportional to amount of resources to backup: more resources -> less sleep
        # Lets say that 1K resources to backup is a reason to select minimum interval, 0 - maximum.
        limit_left = max(self.resource_processed_threshold - counter.resources.operated, 0)
        wait = self.sleep_limits.min + (
            (self.sleep_limits.max - self.sleep_limits.min) * limit_left / self.resource_processed_threshold
        )
        logger.info(
            "Totally checked %d tasks, created %d tasks for %d resources of %s in %.2fs. Next run after %s.",
            counter.tasks.checked, counter.tasks.created,
            counter.resources.operated, common_format.size2str(counter.resources.size << 10),
            time.time() - counter.started, common_format.td2str(wait)
        )
        return wait

    def _check_task(self, tid):
        """
        Try to delete a task if more than BAD_TASK_TTL has passed since its creation
        and it's neither executing, nor in a queue, nor in a final state

        :param tid: Task identifier to be checked.
        :return: task object if it's still running, None otherwise
        :rtype: mapping.Task or None
        """
        task = mapping.Task.objects.with_id(tid)
        if not task or task.execution.status in ctt.Status.Group.FINISH:
            return None
        if task.execution.status not in self.OK_STATUSES:
            age = time.time() - calendar.timegm(task.time.updated.timetuple())
            if age >= self.BAD_TASK_TTL * 3600:
                logger.warn(
                    "Task #%s (status: '%s', host: '%s') is too old: %s, deleting it",
                    tid, task.execution.status, task.execution.host, common_format.td2str(age)
                )
                task.delete()
                return None
        return task


class BackupResources(BaseBackupService):
    """ Resources backup scheduler service. """
    # Workflow:
    # * Search for resources without backup on any sandbox storage
    # * Check backup tasks status if any exist for each specific DC
    # * Group resources into chunks of limited size and create tasks to backup them

    task_priority = ctt.Priority(ctt.Priority.Class.BACKGROUND, ctt.Priority.Subclass.HIGH)
    resource_processed_threshold = 1000
    sleep_limits = SleepLimits(1800, 180)

    def __init__(self, *args, **kwargs):
        super(BackupResources, self).__init__(*args, **kwargs)
        self._tick_interval = self.sleep_limits.min

    @property
    def tick_interval(self):
        return self._tick_interval

    def tick(self):
        counter = self.Counter()
        resources_to_backup = list(
            mapping.Resource.clients_insufficient_redundancy(self.sandbox_config.server.storage_hosts)
        )
        if not resources_to_backup:
            wait = dt.timedelta(seconds=self.sleep_limits.max)
            self._model.time.next_run = self._model.time.last_run + wait
            logger.debug("No resources to backup. Next run after %s", common_format.td2str(wait))
            return
        chunks = []
        counter.resources.checked = len(resources_to_backup)
        logger.info("Checking %d resources.", counter.resources.checked)
        for dc, dc_resources in self._group_resources_by_dc(resources_to_backup):
            dc_resources = list(dc_resources)
            dc_logger = common_log.MessageAdapter(logger, "[DC %(dc)s] %(message)s", {"dc": dc})
            dc_logger.debug("Checking %d resources without replication", len(dc_resources))
            to_backup = []
            for tid, task_resources in self._sort_and_group_by(dc_resources, lambda r: r.task):
                task_resources = list(task_resources)
                if tid <= 1:
                    dc_logger.debug("%d resources have no backup task", len(task_resources))
                    to_backup.append(task_resources)
                    continue
                counter.tasks.checked += 1
                task = self._check_task(tid)
                if task:
                    dc_logger.debug(
                        "%d resources are still replicating by task #%s (status: '%s', host: '%s')",
                        len(task_resources), tid, task.execution.status, task.execution.host,
                    )
                else:
                    dc_logger.debug(
                        "%d resources have finished or expired replication task #%s",
                        len(task_resources), tid
                    )
                    to_backup.append(task_resources)

            chunks.append(self._chunker(
                dc_logger,
                list(it.chain.from_iterable(to_backup)),
                {
                    "dc": dc,
                    "validate_backup_task": True,
                    "register_deps": False
                },
                ["AUTOBACKUP", "REGULAR", dc.upper()]
            ))

        for task_id, chunk in self._creator(list(it.chain.from_iterable(chunks))):
            counter.on_task_create(chunk)
            date = dt.datetime.utcnow()
            common_statistics.Signaler().push(dict(
                type=cts.SignalType.BACKUP_RESOURCES,
                timestamp=date,
                date=date,
                task_id=task_id,
                resources_size=chunk.size << 10,
                resources_amount=chunk.amount,
                dc=chunk.parameters["dc"],
                backup_type=BackupType.BACKUP,
            ))

        wait = dt.timedelta(seconds=self._calculate_wait(counter))
        self._model.time.next_run = self._model.time.last_run + wait
        logger.debug("Next run after %s", common_format.td2str(wait))

    @classmethod
    def _group_resources_by_dc(cls, resources):
        host2dc = {
            client.hostname: client.info["system"].get("dc")
            for client in mapping.Client.objects().all()
        }

        def dc_getter(resource):
            return host2dc.get(resource.host, "unk")

        return cls._sort_and_group_by(resources, dc_getter)

    @staticmethod
    def _sort_and_group_by(items, key):
        return it.groupby(sorted(items, key=key), key)


class ReplicateResources(BaseBackupService):
    """ Resources replication between storage hosts scheduler service. """
    # Workflow:
    # * Check active replication tasks (from previous run)
    # * Get resources without redundancy (required replication)
    # * Group resources into chunks of limited size and create tasks to replicate them

    task_priority = ctt.Priority(ctt.Priority.Class.BACKGROUND, ctt.Priority.Subclass.NORMAL)
    resource_processed_threshold = 2500
    sleep_limits = SleepLimits(3600, 600)
    # Maximum allowed host's resources total size for automatic replication in GiB.
    MAX_ALLOWED_RESOURCES_SIZE = 8096
    MAX_RESOURCE_COUNT_TO_DROP_FROM_HOST = 10000

    def __init__(self, *args, **kwargs):
        super(ReplicateResources, self).__init__(*args, **kwargs)
        self._tick_interval = self.sleep_limits.min

    @property
    def tick_interval(self):
        return self._tick_interval

    def tick(self):
        storages = list(self.sandbox_config.server.storage_hosts)
        with common_context.Timer() as timer:
            with timer[self.replicate_resources.__name__]:
                self.replicate_resources(storages)
            with timer[self.decrease_extra_replication.__name__]:
                self.decrease_extra_replication(storages, timer)
        logger.info("Tick times: %s", timer)

    def replicate_resources(self, storages):
        counter = self.Counter()
        wait_collection = []

        chunks = []
        ctx = self._model.context.get("active_tasks", {})

        for host in random.sample(storages, len(storages)):
            tasks = ctx.get(host, [])
            host_logger = common_log.MessageAdapter(logger, "[%(host)s] %(message)s", {"host": host})
            counter.tasks.checked += len(tasks)
            client = mapping.Client.objects.with_id(host)
            if ctc.Tag.CUSTOM_REPLICATION in client.tags_set:
                host_logger.debug("Host marked for custom replication, skip automatic", host)
                continue

            running = []
            for tid in tasks:
                task = self._check_task(tid)
                if task:
                    host_logger.debug(
                        "Replication task #%s still not completed (status: '%s', host: '%s')",
                        tid, task.execution.status, task.execution.host,
                    )
                    running.append(tid)

            ctx[host] = running
            if running:
                wait_collection.append(self.sleep_limits.min)
                continue

            resources = self._storage_insufficient_redundancy(host, storages)
            if not resources:
                wait_collection.append(self.sleep_limits.max)
                host_logger.debug("No resources to replicate")
                continue

            total_size = sum(r.size for r in resources)  # (KiB)
            if total_size > self.MAX_ALLOWED_RESOURCES_SIZE << 20:
                host_logger.warning(
                    "Avoid automatic replication: total size is %s, limit is %s",
                    common_format.size2str(total_size << 10),
                    common_format.size2str(self.MAX_ALLOWED_RESOURCES_SIZE << 30)
                )
                continue

            dc = client.info["system"].get("dc")
            chunks.append(self._chunker(
                host_logger,
                resources,
                {
                    "avoid_dc": dc,
                    "register_deps": False,
                    "source": host,
                    "validate_backup_task": False,
                    "replication": True,
                },
                ["REDUNDANCY", "REGULAR", dc.upper()]
            ))

        for task_id, chunk in self._creator(list(it.chain.from_iterable(chunks))):
            ctx.setdefault(chunk.parameters["source"], []).append(task_id)
            counter.on_task_create(chunk)
            date = dt.datetime.utcnow()
            common_statistics.Signaler().push(dict(
                type=cts.SignalType.BACKUP_RESOURCES,
                date=date,
                timestamp=date,
                task_id=task_id,
                resources_size=chunk.size << 10,
                resources_amount=chunk.amount,
                dc=chunk.parameters["avoid_dc"],
                backup_type=BackupType.REPLICATION,
            ))

        self._model.context["active_tasks"] = ctx  # No need to save it here - it will be saved later by the caller.
        wait_collection.append(self._calculate_wait(counter))
        self._model.time.next_run = (
            self._model.time.last_run + dt.timedelta(seconds=(sum(wait_collection) / len(wait_collection)))
        )

    @classmethod
    def _storage_insufficient_redundancy(cls, host, storages):  # type: (str, List[str]) -> List[ResourceEntry]
        """
        Lists resources for the given storage host, which has redundancy problems, i.e.,
        has insufficient copies on `H@SANDBOX_STORAGE` hosts group.
        Notice: the request can take significant amount of time (up to several minutes).

        :param host:    Host to be checked, if it's None, check not replicated resources in MDS.
        :return:        A list of tuples with resource ID and resource size in KiB.
        """
        hosts = set(storages)
        hosts.discard(host)
        query = mapping.Resource.objects(
            attributes__key=ctr.ServiceAttributes.BACKUP_TASK,
            state=mapping.Resource.State.READY,
            hosts_states__state=mapping.Resource.HostState.State.OK,
            hosts_states__host__nin=hosts,
            hosts_states__host__in=[host],
            mds=None
        ).order_by("+id").fast_scalar("id", "size")
        return [ResourceEntry(*item) for item in query]

    @staticmethod
    def _extra_replication_base_pipeline(host, storages):
        return [
            {"$match": {
                "state": ctr.State.READY,
                "hosts": {
                    "$elemMatch": {
                        "st": ctr.HostState.OK,
                        "h": host,
                    }
                },
            }},
            {"$project": {
                "_id": 1,
                "hosts": 1,
                "mds": {"$cond": [{"$gt": ["$mds", None]}, 1, 0]},
            }},

            {"$unwind": "$hosts"},
            {"$match": {
                "hosts.st": ctr.HostState.OK,
                "hosts.h": {"$in": storages},
            }},
            {"$group": {
                "_id": "$_id",
                "mds": {"$last": "$mds"},
                "storages": {"$sum": 1},
            }},
        ]

    @classmethod
    def _extra_replication_with_mds_pipeline(cls, host, storages):
        pipeline = cls._extra_replication_base_pipeline(host, storages)
        pipeline.extend([
            {"$match": {
                "mds": {"$gte": 1},
                "storages": {"$gte": 2},
            }},
            {"$limit": cls.MAX_RESOURCE_COUNT_TO_DROP_FROM_HOST},
        ])
        return pipeline

    @classmethod
    def _extreme_extra_replication_pipeline(cls, host, storages):
        pipeline = cls._extra_replication_base_pipeline(host, storages)
        pipeline.extend([
            {"$match": {
                "storages": {"$gte": 3},
            }},
            {"$limit": cls.MAX_RESOURCE_COUNT_TO_DROP_FROM_HOST},
        ])
        return pipeline

    @staticmethod
    def mark_resources_to_delete(create_pipeline_function, host, storages, host_logger):
        host_logger.debug("Apply %s", create_pipeline_function.__name__)
        res = mapping.Resource.aggregate(
            create_pipeline_function(host, storages),
            allowDiskUse=True,
        )
        host_logger.debug("Host has %s resources with extra replication", len(res))
        rids = [_["_id"] for _ in res]
        marked = mapping.Resource.objects(
            id__in=rids,
            hosts_states__host=host,
        ).update(
            set__hosts_states__S__state=ctr.HostState.MARK_TO_DELETE,
        )
        host_logger.debug("%s resources marked to delete", marked)

    @classmethod
    def decrease_extra_replication(cls, storages, timer):
        unreliable = list(
            mapping.Client.objects(tags__all=[str(ctc.Tag.STORAGE), str(ctc.Tag.MAINTENANCE)]).scalar("hostname")
        )
        ok_storages = list(set(storages) - set(unreliable))

        with timer["{}-unreliable".format(len(unreliable))]:
            for host in unreliable:
                host_logger = common_log.MessageAdapter(logger, "[%(host)s] %(message)s", {"host": host})
                cls.mark_resources_to_delete(cls._extra_replication_with_mds_pipeline, host, storages, host_logger)
                cls.mark_resources_to_delete(cls._extreme_extra_replication_pipeline, host, storages, host_logger)

        with timer["{}-storages".format(len(ok_storages))]:
            for host in ok_storages:
                host_logger = common_log.MessageAdapter(logger, "[%(host)s] %(message)s", {"host": host})
                cls.mark_resources_to_delete(cls._extra_replication_with_mds_pipeline, host, storages, host_logger)


class BackupResourcesToMds(base.SingletonService):
    tick_interval = 300

    @staticmethod
    def _drop_stalled_locks(now):
        removed = mapping.ResourceForBackup.objects(
            lock__ne=None,
            till__lte=now,
        ).delete()
        logger.info("Removed %s stalled lock(s)", removed)

    @staticmethod
    def _find_resources_to_backup():
        resources_in_queue = {}
        total_size = 0
        for rid, sources, size in mapping.ResourceForBackup.objects(
            read_preference=mapping.ReadPreference.SECONDARY,
        ).fast_scalar("id", "sources", "size"):
            resources_in_queue[rid] = (set(sources), size)
            total_size += size
        logger.info(
            "There are %s resource(s) to backup with total size %s",
            len(resources_in_queue), common_format.size2str(total_size << 10)
        )
        attr_query = mapping.Q(
            attributes__key="backup_task",
        ) | mapping.Q(
            attributes__key="sync_upload_to_mds",
        ) | mapping.Q(
            attributes=mapping.Resource.Attribute(key="ttl", value="inf"),
        )
        resources_to_backup = list(mapping.Resource.objects(
            attr_query,
            state=ctr.State.READY,
            mds=None,
            md5__ne=ctr.EMPTY_FILE_MD5,
            read_preference=mapping.ReadPreference.SECONDARY,
        ).fast_scalar("id", "size", "hosts_states"))
        logger.info(
            "Found %s resource(s) without backup in MDS with total size %s",
            len(resources_to_backup),
            common_format.size2str(sum(size for rid, size, hosts_states in resources_to_backup) << 10)
        )
        new_backups = []
        new_backups_total_size = 0
        for rid, size, hosts_states in resources_to_backup:
            hosts = {item["h"] for item in hosts_states or () if item["st"] == ctr.HostState.OK}
            sources, res_size = resources_in_queue.get(rid, (None, None))
            if sources is None:
                new_backups.append(
                    mapping.ResourceForBackup(
                        id=rid,
                        size=size,
                        sources=hosts,
                    ).to_mongo()
                )
                new_backups_total_size += size
            elif sources != hosts or res_size != size:
                mapping.ResourceForBackup.objects(
                    id=rid,
                ).update(
                    set__size__=size,
                    set__sources=hosts,
                )
        if new_backups:
            logger.info(
                "Adding %s resource(s) to backup with total size %s",
                len(new_backups), common_format.size2str(new_backups_total_size << 10)
            )
            db = mapping.get_connection().get_default_database()
            # noinspection PyProtectedMember
            coll = db[mapping.ResourceForBackup._get_collection_name()]
            try:
                coll.insert_many(new_backups, ordered=False, bypass_document_validation=True)
            except pymongo.errors.BulkWriteError as ex:
                duplicates = [_["op"]["_id"] for _ in ex.details["writeErrors"] if _["code"] == 11000]
                if duplicates:
                    logger.warning("Resources %s are already added for backup", duplicates)
                else:
                    logger.error("Error occurred while inserting to DB: %s", ex)

    def tick(self):
        now = dt.datetime.utcnow()
        self._drop_stalled_locks(now)
        self._find_resources_to_backup()


class ResourceCopyInfo(
    collections.namedtuple(
        "ResourceCopyInfo",
        "id size bucket"
    )
):
    @classmethod
    def from_db(cls, *args):
        args = list(args)
        args[1] <<= 10  # resource size saved in KiB
        return ResourceCopyInfo(*args)


class CopyResourcesToWarehouse(base.SingletonService):
    tick_interval = 900

    PROCESS_POOL_SIZE = 10
    THREAD_POOL_SIZE = 10
    CHUNK_SIZE = 10000

    @staticmethod
    def __copy(workers_pool, jobs, res, deadline):
        copied = 0
        copied_size = 0
        job_id = None
        while job_id is None:
            if dt.datetime.utcnow() >= deadline:
                break
            ready_jobs = workers_pool.ready_jobs()
            for job_id in ready_jobs:
                size = jobs.pop(job_id).size
                # noinspection PyBroadException
                try:
                    if workers_pool.result(job_id):
                        copied += 1
                        copied_size += size
                except Exception:
                    logger.exception("Unexpected error")
            job_id = workers_pool.spawn(res)
            if job_id is not None:
                jobs[job_id] = res
        return job_id, copied, copied_size

    @staticmethod
    def __wait_jobs(workers_pool, jobs):
        copied = 0
        copied_size = 0
        deadline = time.time() + max(len(jobs), 15)
        logger.info("Waiting for complete of %s job(s)", len(jobs))
        while jobs and time.time() < deadline:
            ready_jobs = workers_pool.ready_jobs()
            for job_id in ready_jobs:
                size = jobs.pop(job_id).size
                # noinspection PyBroadException
                try:
                    if workers_pool.result(job_id):
                        copied += 1
                        copied_size += size
                except Exception:
                    logger.exception("Unexpected error")
            if not ready_jobs:
                time.sleep(1)
        return copied, copied_size

    def _copy(self, res, deadline):
        if dt.datetime.utcnow() >= deadline:
            return
        logger.info(
            "Copying resource #%s with size %s from bucket %s to %s",
            res.id, common_format.size2str(res.size), res.bucket, ctr.WAREHOUSE_BUCKET
        )
        # noinspection PyUnresolvedReferences
        try:
            if self.sandbox_config.common.installation == ctm.Installation.PRODUCTION:
                common_mds.S3().copy(res.bucket, ctr.WAREHOUSE_BUCKET, res.id, logger=logger)
            logger.info("Resource #%s copied from bucket %s to %s", res.id, res.bucket, ctr.WAREHOUSE_BUCKET)
        except common_mds.S3.Exception as ex:
            logger.error(
                "Error while coping resource #%s from bucket %s to %s",
                res.id, res.bucket, ctr.WAREHOUSE_BUCKET, ex
            )
            return False
        mapping.Resource.objects.with_id(res.id).update(set__mds__backup_namespace=ctr.WAREHOUSE_BUCKET)
        logger.info("Resource #%s updated in Sandbox", res.id)
        mapping.ResourcesToWarehouse.objects.with_id(res.id).delete()
        logger.info("Resource #%s removed from queue", res.id)
        return True

    def _copy_resources(self):
        title = "{}{}copier".format(
            setproctitle.getproctitle().rsplit(common_os.PROCESS_TITLE_DELIMITER, 1)[0],
            common_os.PROCESS_TITLE_DELIMITER,
        )
        deadline = dt.datetime.utcnow() + dt.timedelta(seconds=self.tick_interval)
        workers_pool = common_os.WorkersPool(
            lambda r: self._copy(r, deadline),
            self.PROCESS_POOL_SIZE, self.THREAD_POOL_SIZE, title=title
        )
        logger.info("Starting workers")
        workers_pool.start()
        total_copied = 0
        total_copied_size = 0
        jobs = {}
        try:
            while dt.datetime.utcnow() < deadline:
                resources_in_progress = {res.id for res in jobs.values()}
                # noinspection PyProtectedMember
                for res in six.moves.map(
                    lambda _: ResourceCopyInfo.from_db(*_),
                    mapping.ResourcesToWarehouse.objects(
                        read_preference=mapping.ReadPreference.SECONDARY
                    ).order_by("id").limit(self.CHUNK_SIZE).fast_scalar(*ResourceCopyInfo._fields)
                ):
                    if res.id in resources_in_progress:
                        continue
                    job_id, copied, copied_size = self.__copy(workers_pool, jobs, res, deadline)
                    total_copied += copied
                    total_copied_size += copied_size
                    if dt.datetime.utcnow() >= deadline:
                        break
                time.sleep(1)
            copied, copied_size = self.__wait_jobs(workers_pool, jobs)
            total_copied += copied
            total_copied_size += copied_size
            logger.info("Stopping workers")
            workers_pool.stop()
        finally:
            logger.info(
                "Copied %s resources with total size %s",
                total_copied, common_format.size2str(total_copied_size)
            )

    def tick(self):
        common_mds.S3.set_s3_logging_level(logging.INFO)
        self._copy_resources()
