import logging
import datetime as dt

from sandbox import common
from sandbox.common import rest
from sandbox.common.types import client as ctc

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.services import base

logger = logging.getLogger(__file__)


class ClientAvailabilityManager(base.WalleSingletonService):

    class ContextField(common.utils.Enum):
        """ This service process context field constants """
        common.utils.Enum.lower_case()

        AUTOMATION = None

    tick_interval = 30

    def update_host_statuses(self):
        broken_hosts_in_walle = self.get_host_names(
            project__in=self.sandbox_config.common.walle.projects,
            status__nin=["ready"]
        )
        prev_broken_hosts = set(self.context.get("broken", []))

        broken_hosts_ids = [controller.Client.id_from_fqdn(_) for _ in broken_hosts_in_walle]
        broken_clients = list(mapping.Client.objects(hostname__in=broken_hosts_ids))

        for client in broken_clients:
            if ctc.Tag.MAINTENANCE not in client.tags:
                controller.Client.update_tags(client, [ctc.Tag.MAINTENANCE], controller.Client.TagsOp.ADD)

        broken_hosts_in_sandbox = {_.fqdn for _ in broken_clients}

        new_broken_hosts = broken_hosts_in_sandbox - prev_broken_hosts
        logger.info("New broken hosts: %s", sorted(new_broken_hosts))

        new_ready_hosts = prev_broken_hosts - broken_hosts_in_walle
        logger.info("New ready hosts: %s", sorted(new_ready_hosts))

        for fqdn in new_ready_hosts:
            client_id = controller.Client.id_from_fqdn(fqdn)
            client = mapping.Client.objects.with_id(client_id)
            if client:
                controller.Client.update_tags(client, [ctc.Tag.MAINTENANCE], controller.Client.TagsOp.REMOVE)
            else:
                logger.warning("Client %s has been deleted from database", client_id)

        self.context["broken"] = sorted(broken_hosts_in_sandbox)

    @property
    def automation_statuses(self):
        for project in self.sandbox_config.common.walle.projects:
            try:
                response = self.walle_client.projects[project].read(
                    fields=["dns_automation.enabled", "healing_automation.enabled"]
                )
                dns_automation_enabled = response.get("dns_automation", {}).get("enabled", False)
                healing_automation_enabled = response.get("healing_automation", {}).get("enabled", False)
                yield project, dict(
                    dns_automation_enabled=dns_automation_enabled,
                    healing_automation_enabled=healing_automation_enabled
                )
            except rest.Client.HTTPError as exc:
                logger.exception("Failed to get automation statuses from Wall-e for %s: %s", project, exc)
                # on API unavailability we report false-negative
                yield project, dict(
                    dns_automation_enabled=False,
                    healing_automation_enabled=False
                )

    def update_automation_info(self):
        automation_statuses = self.context.setdefault(self.ContextField.AUTOMATION, {})
        for project, walle_statuses in self.automation_statuses:
            context_statuses = automation_statuses.setdefault(project, {})
            updated = False
            for key, value in walle_statuses.items():
                if context_statuses.get(key, None) != value:
                    context_statuses[key] = value
                    updated = True
            if updated:
                automation_statuses[project]["update_ts"] = dt.datetime.now()

    def tick(self):
        self.update_host_statuses()
        self.update_automation_info()
