import cPickle
import httplib
import logging

from sandbox import common
import sandbox.common.types.client as ctc

from sandbox.services import base

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

logger = logging.getLogger(__name__)


class ClientProcessor(base.WalleSingletonService):
    REBOOT_COMMANDS = [ctc.ReloadCommand.REBOOT, ctc.ReloadCommand.POWEROFF]
    tick_interval = 120
    MAX_LIMIT = 1000

    def try_apply_command(self, client, command):
        fqdn = client.info.get("system", {}).get("fqdn")
        if fqdn is None:
            logger.error("Host %s has no fqdn -- skip it", client.hostname)
            return

        try:
            if command.command == ctc.ReloadCommand.REBOOT:
                self.walle_client.hosts[fqdn].reboot(
                    reason="Author: {}. Reason: {}".format(command.author, command.comment)
                )
            elif command.command == ctc.ReloadCommand.POWEROFF:
                self.walle_client.hosts[fqdn]["power-off"](
                    reason="Author: {}. Reason: {}".format(command.author, command.comment)
                )
            controller.Client.next_service_command(client, reset=True, request_commands=self.REBOOT_COMMANDS)

        except common.rest.Client.HTTPError as ex:
            logger.info("Skip host %s. Wall-E api call failed with error: %s.", client.hostname, ex)
            try:
                walle_info = self.walle_client.hosts[fqdn].read()
            except common.rest.Client.HTTPError as inner_ex:
                logger.info("Failed to figure out why %s shouldn't be rebooted: %s", client.hostname, inner_ex)
                return
            else:
                status = walle_info.get("status")
                if status != "ready":
                    logger.info(
                        "%s is already taken care of by %r and is in %r status -- cancel pending %s",
                        client.hostname, walle_info.get("status_author"), status, command.command
                    )
                    controller.Client.next_service_command(client, reset=True, request_commands=self.REBOOT_COMMANDS)

    def reboot_clients(self, clients):
        for client in clients:
            if client.alive or ctc.Tag.Group.OSX in client.tags:
                continue
            command = controller.Client.next_service_command(client, request_commands=self.REBOOT_COMMANDS)
            if command is not None:
                self.try_apply_command(client, command)

    def _get_project_owners(self, project):
        owners = self.walle_client.projects[project].read(fields=["owners"])["owners"]
        result = set()

        for owner in owners:
            if owner[0] == "@":
                result.update(controller.Group.staff_group_content(owner[1:]))
            else:
                result.add(owner)

        return result

    @staticmethod
    def _maybe_update_owner(client, owners):
        if owners ^ set(client.info.get("owners", list())):
            logger.info("Update owners for host %s. New owners: %s", client.hostname, list(owners))
            client.info["owners"] = list(owners)
            mapping.Client.objects(hostname=client.hostname).update_one(
                set__context=mapping.Client.context.to_mongo(cPickle.dumps(client.info))
            )

    def rebuild_owners(self, clients):
        clients_dict = {client.hostname: client for client in clients if ctc.Tag.Group.OSX not in client.tags}
        walle_projects_names = self.context.get("projects", {})
        walle_projects = {}
        removed_projects = set()

        for project in walle_projects_names:
            try:
                walle_projects[project] = self._get_project_owners(project)
            except common.rest.Client.HTTPError as ex:
                if ex.response.status_code == httplib.NOT_FOUND:
                    logger.warning("Project %s not found in Wall-E.", project)
                    removed_projects.add(project)
                else:
                    raise

        for project, owners in walle_projects.iteritems():
            hosts = self.get_host_names(project=project)
            has_hosts = False

            for host in hosts:
                client_id = host.split(".", 1)[0]
                client = clients_dict.pop(client_id, None)
                if client is None:
                    continue

                has_hosts = True
                self._maybe_update_owner(client, owners)

            if not has_hosts:
                logger.warning("There are no hosts in Wall-E project %s", project)
                removed_projects.add(project)

        for hostname, client in clients_dict.iteritems():
            try:
                fqdn = client.info.get("system", {}).get("fqdn")
                if fqdn is None:
                    logger.error("Host %s has not fqdn. Skip this host.", client.hostname)
                    continue

                project = self.walle_client.hosts[fqdn].read(fields=["project"])["project"]
                if project not in walle_projects:
                    walle_projects[project] = self._get_project_owners(project)
                self._maybe_update_owner(client, walle_projects[project])

            except common.rest.Client.HTTPError:
                logger.info("Skip host %s.", client.hostname)

        self.context["projects"] = walle_projects.viewkeys() - removed_projects

    def fix_user_tags(self, clients):
        user_tags_to_remove = set()
        user_tags_cache = set()

        for client in clients:
            removed_tags = []
            for tag in client.tags:
                if tag in ctc.Tag.Group.USER:
                    if tag in user_tags_cache:
                        if tag in user_tags_to_remove:
                            removed_tags.append(tag)
                    else:
                        group = mapping.Group.objects(user_tags__name=tag).first()
                        user_tags_cache.add(tag)
                        if group is None:
                            user_tags_to_remove.add(tag)
                            removed_tags.append(tag)
            controller.Client.update_tags(client, removed_tags, controller.Client.TagsOp.REMOVE)

        if user_tags_to_remove:
            logger.info("Removed user tags: %s", ", ".join(user_tags_to_remove))

    def tick(self):
        clients = controller.Client.list()
        self.reboot_clients(clients)
        self.rebuild_owners(clients)
        self.fix_user_tags(clients)
