import logging

import concurrent.futures

from sandbox.services import base
from sandbox.yasandbox import controller

logger = logging.getLogger(__name__)


class GroupSynchronizer(base.SingletonService):
    """
    Synchronizer of groups from external sources
    """

    tick_interval = 60

    def __init__(self, *args, **kwargs):
        super(GroupSynchronizer, self).__init__(*args, **kwargs)
        self._pool = concurrent.futures.ThreadPoolExecutor(max_workers=16)

    def tick(self):
        futures = []
        cache = controller.GroupCache()
        cache.initialize()

        for group in controller.Group.list():
            sources = group.get_sources()
            if not sources:
                continue

            def sync(group_, sources_):
                logger.info(
                    "Synchronizing group %r with sources: %s",
                    group_.name, [(s.source, s.group) for s in sources_]
                )
                try:
                    controller.Group.sync(group_, cache=cache)
                except Exception:
                    logger.exception("Cannot synchronize group %r", group_.name)

            # Send group for synchronization
            futures.append(self._pool.submit(sync, group, sources))

        # Wait for all to complete the tick
        concurrent.futures.wait(futures)

    def on_stop(self):
        # We wait for jobs at the end of tick, so this should not block
        self._pool.shutdown(wait=True)
