import logging
import datetime as dt
import itertools
import collections

import requests
from concurrent import futures

from library.sky import hostresolver

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.deploy import juggler
from sandbox.services.modules.juggler.components import base


logger = logging.getLogger(__name__)
Revision = collections.namedtuple("Revision", "no datetime")

SANDBOX_SERVERS_CTAG = "k@sandbox_server"
REVISION_URL_TEMPLATE = "http://{}:{}/api/v1.0/service/time/current"


def from_ms_timestamp(ts):
    return dt.datetime.utcfromtimestamp(int(ts) / 1000)


def revision_url_for(hostname, serviceapi_port):
    return REVISION_URL_TEMPLATE.format(hostname, serviceapi_port)


class ArcanumHelper(object):
    BASE_URL = "https://a.yandex-team.ru/api"

    def __init__(self, token):
        self.api = common.rest.ThreadLocalCachableClient(self.BASE_URL, auth=token)

    def history(self, path, limit=100):
        return self.api.tree.history.trunk.arcadia[path].read(repo="arc", limit=limit)

    def commit_datetime(self, rev):
        data = self.api.tree.commit[rev].read(repo="arc")
        return from_ms_timestamp(data["date"])


class TasksAutodeploy(base.BaseCheck):
    """
    Compare tasks code revision deployed on servers to that of sandbox/projects in Arcanum.
    It is made with assumption that once a new revision in Arcanum appears,
    it'll be built and deployed to servers in `max_delay` minutes.
    """

    juggler_check = juggler.TasksAutodeploy

    SERVERS_LIST_TTL = 3600
    CI_LINK = "https://nda.ya.ru/t/lk_TJ3j74sZ5vz"

    def __init__(self, arcanum_token, serviceapi_port, max_delay, *args, **kwargs):
        super(TasksAutodeploy, self).__init__(*args, **kwargs)
        self.arcanum = ArcanumHelper(arcanum_token)
        self.serviceapi_port = serviceapi_port
        self.max_delay = dt.timedelta(minutes=max_delay)

    @common.utils.ttl_cache(SERVERS_LIST_TTL)
    def resolve_servers(self):
        return hostresolver.Resolver().resolveHosts(SANDBOX_SERVERS_CTAG)

    @staticmethod
    def query_server(hostname, serviceapi_port):
        url = revision_url_for(hostname, serviceapi_port)
        hostname = hostname.split(".")[0]
        try:
            r = requests.get(url, timeout=30)
        except (requests.HTTPError, requests.ConnectionError, requests.Timeout):
            logger.error("Server %s didn't respond to API request", hostname)
            return None

        revision = r.headers.get(ctm.HTTPHeader.TASKS_REVISION, None)
        return None if revision is None else int(revision)

    def fetch_deployed_revision(self):
        hostnames = self.resolve_servers()
        if not hostnames:
            self.message = "Skynet resolver returned no servers for {}".format(SANDBOX_SERVERS_CTAG)
            return None

        with futures.ThreadPoolExecutor(max_workers=len(hostnames)) as pool:
            revisions = filter(
                None,
                pool.map(self.query_server, hostnames, itertools.repeat(self.serviceapi_port))
            )

        return None if not revisions else min(revisions)

    def fetch_vcs_revisions_info(self):
        data = self.arcanum.history("sandbox/projects")
        return [
            Revision(no=int(commit["revision"]), datetime=from_ms_timestamp(commit["date"]))
            for commit in data
        ] if data else None

    def check(self):
        try:
            deployed_revision = self.fetch_deployed_revision()
            vcs_revisions_info = self.fetch_vcs_revisions_info()
        except common.rest.Client.HTTPError as exc:
            logger.error("Failed to check tasks code autodeploy status: %s", exc)
            return None

        if None in (deployed_revision, vcs_revisions_info):
            self.status = ctm.JugglerCheckStatus.WARNING
        else:
            utcnow = dt.datetime.utcnow()

            # Are there commits in VCS that must have been already deployed, but have been not?
            # If yes, tasks code autodeploy is broken.
            trunk_has_undeployed_revision = False
            for revision in vcs_revisions_info:
                if revision.datetime >= utcnow - self.max_delay:
                    continue

                if revision.no > deployed_revision:
                    trunk_has_undeployed_revision = True
                break
            else:
                trunk_has_undeployed_revision = True

            deployed_revision_datetime = self.arcanum.commit_datetime(deployed_revision)
            autodeploy_delayed = deployed_revision_datetime < utcnow - self.max_delay

            self.status = (
                ctm.JugglerCheckStatus.CRITICAL
                if autodeploy_delayed and trunk_has_undeployed_revision else
                ctm.JugglerCheckStatus.OK
            )
            self.message = "min(servers): r{} | VCS: r{}. See CI {}".format(
                deployed_revision,
                vcs_revisions_info[0].no,
                self.CI_LINK
            )

        self.send_to_juggler()
        return self.status
