import httplib
import logging
import smtplib
import datetime as dt

from email import header
from email.mime import text
from email.mime import multipart

import sandbox.common.types.misc as ctm
import sandbox.common.types.scheduler as cts
import sandbox.common.types.statistics as ctst
import sandbox.common.types.notification as ctn

from sandbox import common
from sandbox.web import helpers
from sandbox.services import base
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


logger = logging.getLogger(__name__)


class Mailman(base.SingletonService):
    """
    Sends notifications via email or telegram
    """

    DOMAIN = "@yandex-team.ru"
    MAILING_LIST = "sandbox"
    MESSAGE_SIZE_LIMIT = 10240000  # postconf -d message_size_limit
    WARNING_TEMPLATE = (
        "Contents of the notification were removed by Sandbox, as its total size, {}, exceeds or is equal to {}."
    )
    SECURITY_BOTS = {"AshToolsBot", "TashaToolsBot", "WorfToolsBot", "TuvokToolsBot", "PavelToolsBot"}
    SECURITY_BOT_ID = 793835103
    tick_interval = 10
    JUGGLER_BATCH_SIZE = 100

    class StopMailman(Exception):
        """
        Raise this exception to stop Mailman service
        """

    def __init__(self, *args, **kwargs):
        super(Mailman, self).__init__(*args, **kwargs)
        self.zk_name = type(self).__name__
        self._rest_client = common.rest.Client(component=ctm.Component.SERVICE)

        self._node = self.sandbox_config.this.id
        self._telegram_bot_model = None
        self._tasks_owners = {}

        self._handlers = {
            ctn.Transport.EMAIL: self.send_email_notification,
            ctn.Transport.TELEGRAM: self.send_telegram_notification,
            ctn.Transport.Q: self.send_messengerq_notification,
            ctn.Transport.JUGGLER: self.send_juggler_notification,
        }

    def tick(self):
        """
        Send notifications via email or telegram, then calculate tick interval due to amount
        of sent notifications. Formula is pretty simple: if 0 notifications were sent, than
        tick interval is 60 secs, if 10 or more notifications were sent, than tick interval is 1 sec;
        between these values tick interval changes evenly.
        """

        self._telegram_bot_model = mapping.Service.objects(name="telegram_bot").first()
        if not self._telegram_bot_model:
            logger.error("Telegram model not found")

        self._tasks_owners = self.get_tasks_owners()

        sent_messages_cnt = 0
        # noinspection PyTypeChecker
        for transport in ctn.Transport:
            if self._stop_requested.is_set():
                break
            try:
                sent_messages_cnt += self.send_new_notifications(transport)
            except self.StopMailman as e:
                logger.exception("Notification sending error: %s", e)
                self.request_stop()

    def null_logger(self, name):
        null_logger = logging.getLogger(name)
        null_logger.handlers = [logging.NullHandler()]
        null_logger.propagate = False
        return null_logger

    @common.utils.singleton_property
    def telegram_client(self):
        token = common.utils.read_settings_value_from_file(
            self.sandbox_config.server.services.get("telegram_bot", {}).get("telegram", {}).get("token")
        )
        if token is None:
            logger.warning("Telegram token not found")
            return None

        return common.telegram.TelegramBot(token, self.null_logger("telegram"))

    @common.utils.singleton_property
    def juggler_client(self):
        token = common.utils.read_settings_value_from_file(
            common.utils.read_settings_value_from_file(self.sandbox_config.server.auth.oauth.token)
        )
        if token is None:
            logger.warning("Juggler token not found")
            return None

        return common.rest.Client(self.service_config["juggler_client_url"], auth=token)

    @common.utils.singleton_property
    def messengerq_client(self):
        token = common.utils.read_settings_value_from_file(
            common.config.Registry().server.services.messengerq_bot.token
        )
        return common.rest.Client(
            common.config.Registry().server.services.messengerq_bot.url, auth=common.auth.GroupOAuth(token),
            logger=self.null_logger("messengerq")
        )

    def can_send_notification(self, task):
        whitelist = self.service_config["whitelist"]
        if whitelist is None:
            return True

        return task is not None and task.type in whitelist

    def send_new_notifications(self, transport):
        """
        Send new 'transport' notifications.
        :raises StopMailman: error occurred during sending
        :return: number of sent notifications
        """

        try:
            sender = self._handlers[transport]
        except KeyError:
            raise self.StopMailman(
                "Wrong transport param: '{}'. Expecting one of {!r}".format(transport, list(ctn.Transport))
            )

        send_list = list(mapping.Notification.objects(sent=False, transport=transport))
        if not send_list:
            return 0

        success_cnt = 0
        now = dt.datetime.utcnow()

        if transport == ctn.Transport.JUGGLER:
            res = sender(send_list)
            success_cnt += res
        else:
            for notification in send_list:
                # noinspection PyBroadException
                try:
                    res = sender(notification)
                except Exception:
                    logger.exception("Sending notification error #%s", notification.id)
                    res = False

                common.statistics.Signaler().push(dict(
                    type=ctst.SignalType.NOTIFICATION,
                    date=now,
                    timestamp=now,
                    author=notification.author,
                    recipients=notification.send_to,
                    transport=transport,
                    sent=res,
                    task_id=notification.task_id
                ))

                if not res:
                    if not notification.inconsistent:
                        controller.Notification.report_service_error(
                            comment="Can not send notification #{}".format(notification.id),
                            thread_name=self.__class__.__name__
                        )
                        notification.inconsistent = True
                        notification.save()
                    continue
                success_cnt += 1

        logger.info("Sent %d, errors in %d notification(s)", success_cnt, (len(send_list) - success_cnt))
        return success_cnt

    @classmethod
    def mark_notification_as_sent(cls, notification_id):
        mapping.Notification.objects(id=notification_id).update(set__sent=True)

    def _prepare_addresses_to_send(self, addresses):
        if self.service_config["notify_real_users_only"]:
            addresses = filter(lambda addr: controller.user.User.valid(addr) is not None, addresses)

        return [addr + self.DOMAIN for addr in addresses]

    def _generate_report_notification(self, notification, task):
        """
        Generate an HTML email report about the task execution

        :param notification: notification mapping document
        :return: body of multipart HTML email
        :rtype: MIMEMultipart
        """
        # Import at global level can lead to a deadlock in Brigadier
        import bleach

        view_handler = {
            ctn.View.EXECUTION_REPORT: {
                "info_provider": controller.Notification.task_execution_info,
                "template_path": "sandbox/web/templates/notifications/execution_report.html"
            },
            ctn.View.RELEASE_REPORT: {
                "info_provider": controller.Notification.task_release_info,
                "template_path": "sandbox/web/templates/notifications/release_report.html"
            }
        }
        try:
            info = view_handler[str(notification.view)]["info_provider"](task=task, notification=notification)
        except KeyError as e:
            raise self.StopMailman(
                "Error while processing notification #{} - incorrect notification view param '{}' ({}). "
                "Correct values: {!r}".format(notification.id, notification.view, str(e), view_handler.keys()),
            )
        if not info:
            return

        msg_root = multipart.MIMEMultipart("related")
        msg_root.preamble = "This is a multi-part message in MIME format."
        msg_alternative = multipart.MIMEMultipart("alternative")
        msg_root.attach(msg_alternative)
        html_part = text.MIMEText(
            common.utils.force_unicode_safe(
                helpers.draw_template(view_handler[notification.view]["template_path"], info)
            ),
            "html",
            "utf-8"
        )

        text_part = text.MIMEText(
            common.utils.force_unicode_safe(bleach.clean(notification.body, tags=[], strip=True)),
            "plain",
            "utf-8"
        )
        msg_alternative.attach(text_part)
        msg_alternative.attach(html_part)
        return msg_root

    def get_tasks_owners(self):
        result = {}

        # noinspection PyBroadException
        try:
            for task in self._rest_client.suggest.task.read():
                result[task["type"]] = task.get("owners", []) or []
        except common.rest.Client.HTTPError as e:
            logger.error("Unable to get tasks: {}".format(e))

        return result

    def send_email_notification(self, notification):
        mail_type = notification.type
        task = None
        if notification.task_id:
            task = mapping.Task.objects.with_id(notification.task_id)
            if task is None:
                logger.warning("Task #%s does not exist", notification.task_id)
        if task and notification.view != ctn.View.DEFAULT:
            msg = self._generate_report_notification(notification, task)
        else:
            try:
                msg = text.MIMEText(common.utils.force_unicode_safe(notification.body), mail_type, "utf-8")
            except KeyError:
                raise self.StopMailman("Unexpected type param {}".format(mail_type))

        addresses = self.service_config["sender"]
        sender = addresses["urgent"] if notification.urgent else addresses["regular"]
        sender_name = "{} <{}>".format(sender.split("@")[0], sender)

        send_to_str = ", ".join(self._prepare_addresses_to_send(notification.send_to))
        headers = map(lambda kv: u"{}: {}".format(*kv), (
            (ctm.EmailHeader.FROM, sender),
            (ctm.EmailHeader.TO, send_to_str),
            (ctm.EmailHeader.SUBJECT, notification.subject),
            (ctm.EmailHeader.SANDBOX_HOST, self._node),
            (ctm.EmailHeader.SANDBOX_NOTIFICATION_ID, notification.id),
        ))

        if task:
            headers.extend(map(lambda kv: u"{}: {}".format(*kv), (
                (ctm.EmailHeader.SANDBOX_TASK_TYPE, task.type),
                (ctm.EmailHeader.SANDBOX_TASK_OWNER, task.owner),
                (ctm.EmailHeader.SANDBOX_TASK_AUTHOR, task.author),
            )))

            if task.scheduler is not None:
                headers.extend([
                    u"{}: {}".format(ctm.EmailHeader.SANDBOX_SCHEDULER_ID, task.scheduler),
                    u"{}: {}".format(
                        ctm.EmailHeader.SANDBOX_SCHEDULER_TYPE,
                        cts.RunType.SCHEDULED if task.scheduler > 0 else cts.RunType.MANUAL
                    ),
                ])

            if notification.view != ctn.View.DEFAULT:
                owners = self._tasks_owners.get(task.type, [])
                reply_to_str = ", ".join(self._prepare_addresses_to_send([self.MAILING_LIST] + owners))
                headers.append(u"{}: {}".format(ctm.EmailHeader.REPLY_TO, reply_to_str))

        headers.extend(notification.headers)
        for item in headers:
            if ":" not in item:
                item = "{}: ".format(ctm.EmailHeader.SANDBOX_HEADER) + item
            key, value = map(lambda x: x.strip(), item.split(":", 1))
            msg[key] = header.Header(" ".join(value.splitlines()))

        raw_sz = len(msg.as_string())
        if raw_sz >= self.MESSAGE_SIZE_LIMIT:
            sizes = map(common.utils.size2str, (raw_sz, self.MESSAGE_SIZE_LIMIT))
            logger.warning(
                "Size of notification %s (%s) exceeds %s, sending a warning instead.", notification.id, *sizes
            )
            message = self.WARNING_TEMPLATE.format(*sizes)
            msg.set_payload(message)

        # noinspection PyBroadException
        server = smtplib.SMTP("localhost")
        try:
            if self.can_send_notification(task):
                server.sendmail(sender_name, send_to_str.split(", "), msg.as_string())
            else:
                logger.info("Notification %s wasn't sent because it isn't whitelisted.", notification.id)
            self.mark_notification_as_sent(notification_id=notification.id)
        except (smtplib.SMTPException, UnicodeEncodeError):
            logger.exception("Can't send email notification %r. Skip it.", notification.id)
            self.mark_notification_as_sent(notification_id=notification.id)
            return True
        except Exception:
            logger.exception("Can't send email notification %r", notification.id)
            return False
        finally:
            server.quit()
        return True

    @common.utils.ttl_cache(300)
    def check_telegram_chat_group(self, chat_id):
        if not mapping.Group.objects(telegram_chat_id=chat_id).count():
            return False
        result = self.telegram_client.get_chat_member(chat_id, self.SECURITY_BOT_ID)
        if result["ok"] and result["result"]["status"] == "administrator":
            return True
        result = self.telegram_client.get_chat_administrators(chat_id)
        if not result["ok"]:
            return False
        admins = set(user["user"]["username"] for user in result["result"])
        return bool(admins & self.SECURITY_BOTS)

    def send_telegram_notification(self, notification):
        message = common.utils.force_unicode_safe(notification.body)

        if None in (self.telegram_client, self._telegram_bot_model):
            return False

        if notification.task_id:
            task = mapping.Task.objects.with_id(notification.task_id)
            if not self.can_send_notification(task):
                logger.info("Notification %s wasn't sent because it isn't whitelisted.", notification.id)
                self.mark_notification_as_sent(notification_id=notification.id)
                return True

        for username in notification.send_to:
            if controller.User.check_telegram_username(username):
                chat_id = self._telegram_bot_model.context.get("usernames", {}).get(username.lower())
            else:
                chat_id = username if self.check_telegram_chat_group(username) else None

            if not chat_id:
                logger.warning("Chat id for username %s not found.", username)
                continue

            # noinspection PyBroadException
            logger.info("Send notification to user %s with chat_id %s.", username, chat_id)
            try:
                result = self.telegram_client.send_message(chat_id, message, common.telegram.ParseMode.HTML)
                if not result["ok"]:
                    logger.error(
                        "Failed to send message to user %s with chat_id %s. Reason: %s",
                        username, chat_id, result["description"]
                    )
            except:
                logger.exception("Can't send telegram notification %r for user %s", notification.id, username)

        self.mark_notification_as_sent(notification_id=notification.id)
        return True

    def send_messengerq_notification(self, notification):
        message = common.utils.force_unicode_safe(notification.body)

        if self.messengerq_client is None:
            return False

        for chat_id in notification.send_to:
            try:
                self.messengerq_client.sendMessage.create(chat_id=chat_id, text=message)
            except Exception:
                logger.exception("Can't send q notification %r to chat %s", notification.id, chat_id)

        self.mark_notification_as_sent(notification_id=notification.id)
        return True

    def send_events(self, events):
        try:
            result = self.juggler_client.events(events=[_[0] for _ in events])
        except common.rest.Client.HTTPError:
            logger.exception("Can't sent juggler notifications")
            return 0
        mapping.Notification.objects(id__in=[_[1] for _ in events]).update(set__sent=True)

        send_count = 0
        for idx, event in enumerate(result["events"]):
            if event.get("code") != httplib.OK:
                logger.info(
                    "Error on notification to host=%s&service=%s. Error: %s",
                    events[idx][0]["host"], events[idx][0]["service"], event.get("error")
                )
            else:
                send_count += 1

        logger.info("Success sent %s of %s juggler events", send_count, len(events))
        return send_count

    def send_juggler_notification(self, notifications):
        events = []
        send_count = 0

        for notification in notifications:
            for recipient in notification.send_to:
                try:
                    host, service = controller.Notification.perform_juggler_query(recipient)
                    events.append((dict(
                        description=notification.body,
                        host=host,
                        service=service,
                        status=notification.check_status,
                        tags=notification.juggler_tags
                    ), notification.id))
                    if len(events) >= self.JUGGLER_BATCH_SIZE:
                        send_count += self.send_events(events)
                        events = []
                except:
                    logger.exception("Can't process juggler notification")

        if events:
            send_count += self.send_events(events)
        return send_count
