import logging
import datetime as dt
import multiprocessing

from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import patterns as common_patterns

from sandbox.common.types import resource as ctr

from sandbox.services import base
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

logger = logging.getLogger(__name__)


class Bucket(object):
    STAT_UPDATE_INTERVAL = 1800  # in seconds
    CHECK_CLEANUP_INTERVAL = 900  # in seconds
    CLEANUP_CHUNK_SIZE = 5000
    GUARANTEE_TTL = dt.timedelta(days=1)

    def __init__(self, state, check_stop, doc):
        self.__state = state
        self.__check_stop = check_stop
        self.__doc = doc
        self.__name = doc.name
        self.__update()

    @common_patterns.singleton_classproperty
    def __s3_idm(self):
        mds_settings = common_config.Registry().common.mds
        return common_rest.Client(base_url=mds_settings.s3.idm.url)

    def __update(self):
        now = dt.datetime.utcnow()
        update_threshold = now - dt.timedelta(seconds=self.STAT_UPDATE_INTERVAL)
        if self.__doc.updated_at is not None and self.__doc.updated_at > update_threshold:
            return
        # noinspection PyBroadException
        try:
            data = self.__s3_idm.stats.buckets[self.__name][:]
            max_size = data.get("max_size")
            used_space = data.get("used_space")
            if max_size is None:
                logger.warning("max_size for bucket '%s' is not set, skipping", self.__name)
                return
            free_space = max_size - used_space
            self.__doc.total = max_size
            self.__doc.free = free_space
            self.__doc.updated_at = now
            self.__doc.save()
        except Exception:
            logger.exception("Error while updating bucket info")

    def __space_to_cleanup(self):
        self.__check()
        free_space = self.__doc.free + self.__state["size"]
        if self.__doc.lru_threshold is None:
            return 0
        return max(self.__doc.lru_threshold - free_space, 0)

    def __check(self):
        now = dt.datetime.utcnow()
        check_timestamp = self.__state.get("timestamp")
        if check_timestamp is not None and (now - check_timestamp).total_seconds() < self.CHECK_CLEANUP_INTERVAL:
            return
        pipeline = [
            {"$match": {
                "mds.n": self.__name,
                "attrs.k": {"$ne": "ttl"},
                "fc": True,
            }},
            {"$project": {
                "size": 1,
            }},
            {"$group": {
                "_id": None,
                "size": {"$sum": "$size"},
                "count": {"$sum": 1},
            }},
        ]
        size = 0
        count = 0
        with mapping.switch_db(mapping.Resource, mapping.ReadPreference.SECONDARY) as Resource:
            for item in Resource.aggregate(pipeline, allowDiskUse=True):
                size = item["size"]
                count = item["count"]
                break
        size <<= 10
        logger.info(
            "%s resource(s) with total size %s currently marked to delete in bucket %s",
            count, common_format.size2str(size), self.__name
        )
        self.__state.update(timestamp=now, size=size, count=count)

    def cleanup(self):
        space_to_cleanup = self.__space_to_cleanup()
        if not space_to_cleanup:
            return 0
        if self.__check_stop():
            return 0
        logger.info(
            "Starting cleanup of bucket %s, need to remove extra %s",
            self.__name, common_format.size2str(space_to_cleanup)
        )
        total_size = 0
        rids = []
        guarantee_survival_time = dt.datetime.utcnow() - self.GUARANTEE_TTL
        for rid, size in mapping.Resource.objects(
            mds__namespace=self.__name,
            attributes__key__ne="ttl",
            force_cleanup__ne=True,
            time__accessed__lt=guarantee_survival_time,
            read_preference=mapping.ReadPreference.SECONDARY,
        ).order_by("+time__accessed").fast_scalar("id", "size").limit(self.CLEANUP_CHUNK_SIZE):
            total_size += size << 10
            rids.append(rid)
            if self.__check_stop() or total_size >= space_to_cleanup:
                break
        if rids:
            logger.info(
                "Going to delete %s resource(s) with total size %s in bucket %s: %s",
                len(rids), common_format.size2str(total_size), self.__name, rids
            )
            controller.Resource.list_resources_audit(
                "Mark resource as deleted (LRU)", rids, state=ctr.State.DELETED
            )
            updated = mapping.Resource.objects(id__in=rids).update(
                set__state=ctr.State.DELETED,
                set__time__updated=dt.datetime.utcnow(),
                set__force_cleanup=True,
            )
            logger.info("%s resource(s) marked as deleted in bucket %s", updated, self.__name)
            self.__state["size"] += total_size
        return len(rids)


class MdsLruCleaner(base.SingletonService):
    MAX_WORKERS = 10

    tick_interval = 1800

    def tick(self):
        self.context.setdefault("buckets_states", {})
        total_count = 0
        thread_pool = multiprocessing.pool.ThreadPool(self.MAX_WORKERS)
        results = []
        check_stop = self._stop_requested.is_set
        for doc in mapping.Bucket.objects().order_by("free"):
            bucket_state = dict(self.context["buckets_states"].setdefault(doc.name, {}))
            bucket = Bucket(bucket_state, check_stop, doc)
            results.append((thread_pool.apply_async(bucket.cleanup), bucket_state, doc.name))
        thread_pool.close()
        thread_pool.join()
        for result, bucket_state, bucket_name in results:
            total_count += result.get()
            self.context["buckets_states"][bucket_name] = bucket_state
        if total_count:
            self._model.time.next_run = dt.datetime.utcnow()
