import time
import logging

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from sandbox.services import base as service_base

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping as mp


logger = logging.getLogger(__name__)


class QRequest(common.utils.Enum):
    common.utils.Enum.lower_case()

    TRACK = None  # track task execution statuses
    SANDBOX_SUBSCRIBE_USER = None  # write one message to bot for subscribing user private chat
    SANDBOX_SUBSCRIBE_GROUP = None  # write one message to bot for subscribing group chat


class QMessage(object):
    def __init__(self, message):
        message_info = message.get("message", {})
        self.text = message_info.get("text")
        self.chat = message_info.get("chat", {}).get("id")
        self.updated_id = message["update_id"]
        uid = message_info.get("from", {}).get("uid")
        self.uid = str(uid) if uid is not None else None

    def __repr__(self):
        return "Message(text={text}, chat={chat}, updated_id={updated_id}, uid={uid})".format(
            text=self.text, chat=self.chat, updated_id=self.updated_id, uid=self.uid
        )


class MessengerQBot(service_base.SingletonService):
    tick_interval = 5

    MESSAGES_CHUNK_SIZE = 100  # number of messages to get by one request

    # response text for task tracking request in wrong format
    WRONG_TASK_TRACKING_REQUEST_REPLY = "Expected request format: /track <task_id> [<status> <status> ...]"

    STAFF_TTL = 300  # how often to fetch the aforementioned group's contents, seconds

    @common.utils.classproperty
    def name(cls):
        return "messengerq_bot"

    @common.utils.singleton_property
    def messengerq_client(self):
        token = common.utils.read_settings_value_from_file(
            common.config.Registry().server.services.messengerq_bot.token
        )
        return common.rest.Client(
            common.config.Registry().server.services.messengerq_bot.url, auth=common.auth.GroupOAuth(token)
        )

    @common.utils.singleton_property
    def staff_token(self):
        return common.utils.read_settings_value_from_file(common.config.Registry().server.auth.oauth.token)

    @common.utils.singleton_property
    def staff_api(self):
        return common.rest.Client(
            base_url=common.config.Registry().server.auth.staff.url,
            auth=self.staff_token,
            logger=logger,
            ua="Sandbox.MessengerQBot"
        )

    def get_messages(self):
        """
        Read incoming messages from messenger Q
        :return: list of QMessage objects
        """
        messages = []
        offset = self.context.get("offset", 0)
        while True:
            params = {"offset": offset}
            chunk = self.messengerq_client.getUpdates.read(**params)
            logger.info("Read message chunk of length %s", len(chunk))
            messages.extend([QMessage(_) for _ in chunk])
            if chunk:
                offset = messages[-1].updated_id + 1
            if len(chunk) < self.MESSAGES_CHUNK_SIZE:
                break
        self.context["offset"] = offset
        return messages

    def track_task(self, message):
        """
        Add task tracking

        :param message: QMessage object
        :return: reply messege text
        """
        splited_message = message.text.lower().strip().split()

        if len(splited_message) < 2:
            return self.WRONG_TASK_TRACKING_REQUEST_REPLY

        task_id, statuses = splited_message[1], splited_message[2:]

        try:
            task_id = int(task_id.strip("#"))
        except (TypeError, ValueError):
            return self.WRONG_TASK_TRACKING_REQUEST_REPLY

        task = mp.Task.objects.with_id(task_id)

        if task is None:
            return "Task {} doesn't exist".format(task_id)

        statuses = [status.strip(',').upper() for status in statuses]
        if not statuses:
            statuses = [
                ctt.Status.SUCCESS, ctt.Status.FAILURE,
                ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                ctt.Status.TIMEOUT, ctt.Status.EXPIRED
            ]

        try:
            notification = controller.Notification.notification(
                ctn.Transport.Q,
                statuses,
                [message.chat]
            )

            for task_notification in task.notifications:
                if all(
                    getattr(task_notification, field) == getattr(notification, field)
                    for field in ("transport", "statuses", "recipients")
                ):
                    return "The same notification is already added"

            task.notifications.append(notification)
            mp.Task.objects(id=task_id).update_one(set__notifications=task.notifications)
            controller.TaskStatusNotifierTrigger.append(
                task_id, notification,
                resolver=lambda *x: x
            )
            return "Tracking added"
        except Exception as ex:
            logger.error("Error in appending notification trigger.", exc_info=ex)
            return "Sorry, it was strange database error, try again later or ask sandbox team to help"

    def subscribe_user(self, message):
        """
        Set messenger_chat_id field to mapping.User model to chat_id from messege
        :param message: QMessage object
        :return: reply messege text
        """
        if message.uid is None:
            return "Uid is None, can't subscribe user."

        try:
            login = controller.User.uid_to_login(message.uid)
        except:
            return "Can't find user by uid."

        user = controller.User.validate(login)
        if user:
            mp.User.objects.filter(login=login).update(set__messenger_chat_id=message.chat)

        return "Subscribed."

    def subscribe_group(self, message):
        """
        Do nothing, only for view

        :param message: QMessage object
        :return: reply messege text
        """
        return "Now i know this chat."

    # functions to process message type
    processor_functions = {
        QRequest.TRACK: track_task,
        QRequest.SANDBOX_SUBSCRIBE_USER: subscribe_user,
        QRequest.SANDBOX_SUBSCRIBE_GROUP: subscribe_group,
    }

    def process_simple_message(self, message):
        """
        Parse message and call one of processor_functions and answer for messsages
        :param message: QMessage object
        """
        if message.chat is None:
            logger.error("Chat id is None, very strange. %r", message)
            return

        if message.text is None:
            return

        text = message.text.lower().strip().split()

        if not text:
            return
        method = text[0].lstrip("/")

        if method not in self.processor_functions:
            return

        result = self.processor_functions[method](self, message)
        self.messengerq_client.sendMessage.create(chat_id=message.chat, text=result)

    def process_messages(self):
        """ Read incoming unread messages and process them """
        messages = self.get_messages()
        for message in messages:
            self.process_simple_message(message)

    def tick(self):
        now = time.time()
        try:
            self.process_messages()
        except (Exception, BaseException) as exc:
            logger.error("Unexpected exception has occurred.", exc_info=exc)
        logger.debug("Full cycle took %ds", int(time.time() - now))
