import six
import time
import datetime as dt
import requests

from sandbox.common import fs as common_fs
from sandbox.common import config as common_config
from sandbox.common import format as common_format


class SolomonClient(object):

    def __init__(self, project=None, cluster=None, service=None, common_labels=None, default_time=None):
        self._common_labels = common_labels
        self._default_time = int(time.time()) if default_time is None else self._validate_time(default_time)

        config = common_config.Registry()
        push_settings = config.server.solomon.push

        self._url = config.common.solomon.api + "/push?project={}&cluster={}&service={}".format(
            project or push_settings.project,
            cluster or push_settings.cluster,
            service or push_settings.service,
        )

        self._headers = {"Content-Type": "application/json"}
        token = common_fs.read_settings_value_from_file(config.common.solomon.token)
        if token:
            self._headers["Authorization"] = "OAuth " + token

        self._sensors = []

    @staticmethod
    def _validate_time(t):
        if isinstance(t, float):
            t = int(t)
        if isinstance(t, six.integer_types):
            return t
        if isinstance(t, dt.datetime):
            return common_format.utcdt2iso(t)
        raise ValueError("Unsupported time value: {}".format(t))

    def add_sensor(self, name, value, labels=None, ts=None):
        if labels is None:
            labels = {}
        ts = self._default_time if ts is None else self._validate_time(ts)
        self._sensors.append({
            "labels": dict({"sensor": name}, **labels),
            "ts": ts,
            "value": value,
        })

    def send_data(self, timeout=30):
        if not self._sensors:
            return
        payload = {
            "sensors": self._sensors,
        }
        if self._common_labels is not None:
            payload["commonLabels"] = self._common_labels
        return requests.post(self._url, json=payload, timeout=timeout, headers=self._headers)
