import six
import logging
import itertools
import collections
import datetime as dt
import threading as th

import concurrent.futures

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.scheduler as cts

from sandbox.services import base
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

logger = logging.getLogger(__name__)


class Scheduler(base.ThreadedService):
    """ Checks all schedulers on a periodic basis. """

    notification_timeout = 5
    tick_interval = 30  # default interval
    kamikadze_ttl = 600  # how soon should this process self-destruct if the main thread is halted, seconds

    TICK_INTERVAL_PROCESS_SCHEDULER = 30
    TICK_INTERVAL_OWNER_COHERENCE = 60 * 10
    SANDBOX_ML = "sandbox-dev"

    ABC_MANAGEMENT_SCOPE = "services_management"

    def __init__(self, *args, **kwargs):
        super(Scheduler, self).__init__(*args, **kwargs)

        # for backward compatibility with the old Scheduler to avoid various races
        self.zk_name = type(self).__name__
        self.max_subworkers = 50
        self.pool = concurrent.futures.ThreadPoolExecutor(max_workers=self.max_subworkers)

        self.blink_notification_interval = dt.timedelta(
            seconds=common.config.Registry().server.services.task_status_notifier.notification_blink_interval
        )
        self.notification_interval = dt.timedelta(
            seconds=common.config.Registry().server.services.task_status_notifier.notification_interval
        )

        self._kamikadze = None

    @property
    def targets(self):
        return [
            self.Target(self.scheduler_starter, interval=self.TICK_INTERVAL_PROCESS_SCHEDULER),
            self.Target(self.check_author_to_owner_coherence, interval=self.TICK_INTERVAL_OWNER_COHERENCE),
        ]

    @classmethod
    def head_of_department(cls, group):
        """returns the head of the appropriate to group abc service logins list"""
        head_logins = []
        try:
            abc_service_id = common.abc.sandbox_group_to_abc_id(group)
            head_logins = common.abc.abc_group_content("/" + cls.ABC_MANAGEMENT_SCOPE, id=abc_service_id)
        except common.rest.Client.HTTPError as ex:
            logging.error("Can't get the head of department due to the error: %s", ex.message)
        return head_logins

    @staticmethod
    def send_notification(scheduler, subject, body, add_recipients=None, carbon_copy=None):
        addresses = [scheduler.author, scheduler.owner]
        recipients = controller.Notification.expand_groups_emails(addresses)
        recipients.update(add_recipients or [])

        controller.Notification.save(
            transport=controller.Notification.Transport.EMAIL,
            send_to=recipients,
            send_cc=carbon_copy,
            subject=subject,
            body=body
        )

    @staticmethod
    def _add_mail_footer(body):
        footer = (
            "\n\nThis message is automatically generated by Sandbox."
            "\nReply to this mail (or directly @{sandbox_dev}) if you have any questions."
        ).format(sandbox_dev=Scheduler.SANDBOX_ML)
        return body + footer

    @staticmethod
    def _scheduler_stopped_message(scheduler_id, error_desc):
        """
        :return tuple: email subject and body
        """
        body = (
            "Scheduler #{id} ({link}) has been stopped\n"
            "The reason: {error_desc}"
        ).format(
            id=scheduler_id,
            link=common.urls.get_scheduler_link(scheduler_id),
            error_desc=error_desc,
        )

        return "Scheduler #{} has been stopped".format(scheduler_id), Scheduler._add_mail_footer(body)

    @staticmethod
    def _revival_message(scheduler):
        """
        :return tuple: email subject and body
        """

        body = (
            "Scheduler #{id} ({link}) has successfully started the task after one or multiple fails."
        ).format(
            id=scheduler.id,
            link=common.urls.get_scheduler_link(scheduler.id),
        )

        subject = "Scheduler's #{} task has been successfully started after fails".format(
            scheduler.id
        )
        return subject, Scheduler._add_mail_footer(body)

    @staticmethod
    def _task_fail_message(scheduler):
        """
        :return tuple: email subject and body
        """

        comment = controller.Scheduler.get_notification(scheduler)[2]
        body = (
            "Scheduler's #{sched_id} ({link}) task is falling with notification:\n"
            "{comment}\n"
            "Next notification is scheduled after {notification_time}"
        ).format(
            sched_id=scheduler.id,
            comment=comment,
            link=common.urls.get_scheduler_link(scheduler.id),
            notification_time=scheduler.last_notification_time
        )

        subject = "Scheduler's #{} task is constantly falling".format(scheduler.id),
        return subject, Scheduler._add_mail_footer(body)

    def scheduler_starter(self):
        logger.info("Start checking schedulers")

        if mapping.Scheduler._get_collection() is None:
            logger.exception("Scheduler collection is None. Check aborted.")
            return

        if not self._kamikadze:
            self._kamikadze = common.threading.KamikadzeThread(self.kamikadze_ttl, logger)
            self._kamikadze.start()

        logger.info("Check tasks spawned by schedulers")
        schedulers_with_finished_tasks = self._get_schedulers_finished_tasks()

        logger.info("Check ready-to-run schedulers")
        ready_ids = controller.Scheduler.list_query(
            ready_at=dt.datetime.utcnow(),
            status=(cts.Status.WAITING, cts.Status.WATCHING),
            scalar=["id"]
        )
        for scheduler_id in ready_ids:
            schedulers_with_finished_tasks.setdefault(scheduler_id, [])

        schedulers_with_check_time = mapping.Scheduler.objects(
            id__in=schedulers_with_finished_tasks.keys()
        ).fast_scalar("id", "time__checked")
        logger.info("There are %s schedulers to check", len(schedulers_with_check_time))

        # schedulers are sorted by last check date (never to now), then by creation date (newest to oldest)
        futures = [
            self.pool.submit(self._process_scheduler, scheduler_id, schedulers_with_finished_tasks[scheduler_id])
            for scheduler_id, _ in sorted(schedulers_with_check_time, key=lambda sch: (bool(sch[1]), sch[1], -sch[0]))
        ]
        concurrent.futures.wait(futures)
        self._kamikadze.ttl = self.kamikadze_ttl

        return "checked all schedulers"

    @staticmethod
    def _last_successful_task_run(scheduler):
        return mapping.Task.objects(scheduler=scheduler.id, execution__status__in=(
            ctt.Status.SUCCESS, ctt.Status.RELEASED, ctt.Status.RELEASING, ctt.Status.NOT_RELEASED
        )).fast_scalar("time__updated").order_by("-id").first()

    @staticmethod
    def _last_failed_task_run(scheduler):
        return mapping.Task.objects(
            scheduler=scheduler.id, execution__status__in=(ctt.Status.FAILURE, ctt.Status.EXCEPTION)
        ).fast_scalar("time__updated").order_by("-id").first()

    def _monitor_task_on_successful_start(self, scheduler):
        now = dt.datetime.utcnow()
        notification_time = scheduler.last_notification_time
        last_failed_run = self._last_failed_task_run(scheduler) or now

        if last_failed_run + self.blink_notification_interval < now < notification_time:
            mapping.Scheduler.objects(id=scheduler.id).update_one(
                set__last_notification_time=now,
            )
            subject, body = self._revival_message(scheduler)
            try:
                logger.debug("sending successful scheduler %s task start notification", scheduler.id)
                self.send_notification(scheduler, subject, body)
            except Exception:
                logging.exception("Unable to send scheduler %s notification", scheduler.id)

    def _monitor_task_on_failed_start(self, scheduler):
        now = dt.datetime.utcnow()
        notification_time = scheduler.last_notification_time
        last_successful_run = self._last_successful_task_run(scheduler) or dt.datetime.min

        if now - last_successful_run > self.blink_notification_interval and notification_time < now:
            notification_time = now + self.notification_interval
            mapping.Scheduler.objects(id=scheduler.id).update_one(
                set__last_notification_time=notification_time
            )
            subject, body = self._task_fail_message(scheduler)
            try:
                logger.debug("sending failed scheduler %s task start notification", scheduler.id)
                self.send_notification(scheduler, subject, body, carbon_copy="sandbox-errors")
            except Exception:
                logging.exception("Unable to send scheduler %s notification", scheduler.id)

    def _process_scheduler(self, scheduler_id, finished_tasks_ids=None):
        if self._stop_requested.is_set():
            return
        myno = th.current_thread().name
        scheduler = None
        try:
            scheduler = controller.Scheduler.load(scheduler_id)
            delta = (
                common.utils.td2str(dt.datetime.utcnow() - scheduler.time.checked) + " ago"
                if scheduler.time.checked else "never"
            )
            logger.info("[%s] Check scheduler #%s (checked: %s)", myno, scheduler_id, delta)
            controller.Scheduler.check(scheduler)
            if finished_tasks_ids:
                mapping.Scheduler.objects.with_id(scheduler_id).update(pull_all__cur_tasks_ids=finished_tasks_ids)
            self._monitor_task_on_successful_start(scheduler)

        except (mapping.OperationError, mapping.OperationFailure):
            logger.exception("[%s] Database error detected")
            return

        except Exception as ex:
            # workaround bug in mongoengine `_collection == None`
            if isinstance(ex, AttributeError) and ex.message == "'NoneType' object has no attribute 'find'":
                logger.exception(
                    "Probably mongoengine bug '_collection is None'. Skip scheduler #%s",
                    scheduler_id
                )
                return

            if scheduler.status == cts.Status.FAILURE:
                return

            logger.exception("Unable to run scheduler #%s", scheduler_id)
            self._monitor_task_on_failed_start(scheduler)

    def check_author_to_owner_coherence(self):
        logger.info("Checking schedulers' authors with unmatched group")

        if mapping.Scheduler._get_collection() is None:
            logger.exception("Scheduler collection is None. Check aborted.")
            return

        fields = ("id", "author", "owner")
        SchedulerModel = collections.namedtuple("Scheduler", fields)
        scheduler_by_id = dict()
        for fs in mapping.Scheduler.objects(status__in=(cts.Status.WATCHING, cts.Status.WAITING)).fast_scalar(*fields):
            scheduler = SchedulerModel(*fs)
            scheduler_by_id[scheduler.id] = scheduler

        coherence = controller.Scheduler.check_owner_coherence(scheduler_by_id.values())
        to_be_stopped = [(id_, error_desc) for id_, (ok, error_desc) in six.iteritems(coherence) if not ok]

        stopped_schedulers = mapping.Scheduler.objects(id__in=[s[0] for s in to_be_stopped])
        for sched in stopped_schedulers:
            scheduler_by_id[sched.id] = sched

        for scheduler_id, error_desc in to_be_stopped:
            scheduler = scheduler_by_id[scheduler_id]
            controller.Scheduler.set_status(scheduler, cts.Status.STOPPED)
            scheduler.save()

            logger.info("%s for the scheduler #%s", error_desc, scheduler_id)
            subject, body = self._scheduler_stopped_message(scheduler_id, error_desc)
            notifications = []
            if scheduler.scheduler_notifications:
                notifications = scheduler.scheduler_notifications.notifications
            self.send_notification(
                scheduler,
                subject, body,
                add_recipients=list(itertools.chain.from_iterable(
                    (n.resolved_recipients for n in notifications if cts.Status.STOPPED in n.statuses)
                )),
                carbon_copy=[self.SANDBOX_ML] + self.head_of_department(group=scheduler.owner)
            )
        if to_be_stopped:
            return "Schedulers {} were stopped since authors have unmatched group".format(
                list(s[0] for s in to_be_stopped)
            )
        return "All schedulers checked and none of them have unmatched group"

    @staticmethod
    def _get_schedulers_finished_tasks():
        tasks_ids = []
        for cur_tasks_ids in mapping.Scheduler.objects(status__nin=["DELETED", "STOPPED"]).fast_scalar("cur_tasks_ids"):
            if cur_tasks_ids:
                tasks_ids.extend(cur_tasks_ids)
        tasks = mapping.Task.objects(id__in=tasks_ids).fast_scalar("id", "execution__status", "scheduler")
        scheduler_finished_tasks = collections.defaultdict(list)
        for task_id, task_status, scheduler_id in tasks:
            if scheduler_id and task_status in set(ctt.Status.Group.FINISH) | set(ctt.Status.Group.BREAK):
                scheduler_finished_tasks[scheduler_id].append(task_id)
        logger.debug("Finished tasks: %s", list(itertools.chain.from_iterable(scheduler_finished_tasks.values())))
        return scheduler_finished_tasks

    def on_stop(self):
        super(Scheduler, self).on_stop()
        self.pool.shutdown(wait=True)
        self._kamikadze.stop()
        self._kamikadze = None
