import os
import logging
import cachetools

from sandbox import common
from sandbox.services.modules.statistics_processor.processors import base
from sandbox.services.modules.statistics_processor.schemas import yt_schemas

logger = logging.getLogger(__name__)


class YTProcessor(base.Processor):
    """
    Push signals to YT tables
    """

    TABLE_NAME_PATTERN = "%Y-%m-%d"
    YT_CLIENTS = {}
    BATCH_SZ = 50000

    TABLES_TO_REMEMBER = 1024
    CREATED_TABLES = cachetools.LRUCache(TABLES_TO_REMEMBER)

    @staticmethod
    def date_extractor(signal):
        date_ = signal.get(base.Processor.TIMESTAMP_FIELD)
        if date_ is not None:
            date_ = date_.strftime(YTProcessor.TABLE_NAME_PATTERN)
        return date_

    @common.utils.singleton_classproperty
    def yt_token(_):
        return common.utils.read_settings_value_from_file(common.config.Registry().common.yt.token)

    @classmethod
    def initialize(cls, _):
        pass

    @classmethod
    def yt_client(cls, proxy):
        import yt.wrapper as yt

        return cls.YT_CLIENTS.setdefault(proxy, yt.YtClient(proxy=proxy, token=cls.yt_token))

    def process(self, signals, signal_type, logger_, timer):
        import yt.wrapper as yt

        if signal_type not in yt_schemas.YT_MODELS:
            logger.warning("Unknown YT signal %s", signal_type)
            return

        with timer["grouping"]:
            by_date = {}
            for signal in signals:
                by_date.setdefault(self.date_extractor(signal), []).append(signal)
                signal[self.TIMESTAMP_FIELD] = base.DATETIME_CONVERTER.encode(signal[self.TIMESTAMP_FIELD])

        target = yt_schemas.YT_MODELS[signal_type]
        client = self.yt_client(target.meta.proxy)

        with timer["insertion"]:
            for table_name, signals in by_date.iteritems():
                path = os.path.join(target.meta.path, table_name).format(
                    environment=common.config.Registry().common.installation.lower(),
                )
                try:
                    table = yt.TablePath(name=path, append=True)
                    if path not in self.CREATED_TABLES:
                        client.create(
                            "table", table, recursive=True, ignore_existing=True,
                            attributes=dict(schema=target.schema)
                        )
                        self.CREATED_TABLES[path] = True

                    client.write_table(table, signals, format=yt.JsonFormat())

                except yt.YtHttpResponseError as exc:
                    logger.error("Failed to insert %d signal(s) into %s: %r", len(signals), path, exc.error)
