# WARNING: if you're gonna import this module on the top level of a forking process, do not.
# Some of infi.clickhouse_orm submodules construct a top-level logger,
# which typically results in a deadlock.

# Now back to the good part!
# Declare your signal classes, derived from, for example, infi.clickhouse_orm.models.Model, here,
# and add them to a dictionary below

import inspect

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.common.types.database as ctd
import sandbox.common.types.statistics as cts

from sandbox.yasandbox.database.clickhouse import fields
from sandbox.yasandbox.database.clickhouse import models


class ApiCall(models.DistributedModel):
    schema_version = 12

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # request date and time (UNIX timestamp)
    user = fields.StringField()  # login of the request author
    method = fields.StringField()  # name of the method; may contain spaces
    source = fields.Enum8Field(fields.wrap_enum(ctt.RequestSource))  # type of the request (from GUI, task, ...)
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the request
    read_preference = fields.Enum8Field(fields.wrap_enum(ctd.ReadPreference))  # database read preference
    response_code = fields.Int16Field()  # HTTP response code

    # Request execution metrics, durations are milliseconds
    duration = fields.Int32Field()  # total request duration
    mongodb_duration = fields.Int32Field(default=0)  # MongoDB operations duration
    serviceq_duration = fields.Int32Field(default=0)  # ServiceQ operations duration

    reqid = fields.StringField()  # request identifier
    task_id = fields.NullableField(fields.Int64Field())  # identifier of a task which sent the request (may be null)
    query_string = fields.StringField()  # a string of "key=value&another_key=another_value" format; keys are sorted
    #: name of entity this request's consumed time is attributed to
    quota_owner = fields.NullableField(fields.StringField())

    #: total request duration from the client's point of view, milliseconds, INCLUDING last shot's duration
    total_duration = fields.Int32Field()
    retry_reason = fields.NullableField(fields.Int32Field())  # last error code that caused the retry, if applicable

    #: time spend between ServiceApi and Legacy Web Server
    proxy_duration = fields.Int32Field(default=0)  # the duration of proxy request from ServiceApi to Legacy Web Server
    processing_duration = fields.Int32Field(default=0)  # the duration of processing request by Legacy Web Server


class ApiCallClientSide(models.DistributedModel):
    schema_version = 3

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # request date and time (UNIX timestamp)
    task_id = fields.Int64Field()  # identifier of a task which sent the request
    client_id = fields.LowCardinalityField(fields.StringField())  # id of a client from which the request was sent
    method = fields.LowCardinalityField(fields.StringField())  # HTTP method name
    response_code = fields.NullableField(fields.Int16Field())  # HTTP response code, if the request was performed
    duration = fields.Int32Field()  # total request duration, milliseconds
    reqid = fields.StringField()  # request identifier
    query_string = fields.StringField()  # a string of "key=value&another_key=another_value" format; keys are sorted
    path = fields.StringField()  # HTTP path relative to Sandbox root, like /api/v1.0/task/1234567


class ApiUsage(models.DistributedModel):
    schema_version = 2

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "login")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # method call date and time (UNIX timestamp)
    login = fields.LowCardinalityField(fields.StringField())  # user login
    quota = fields.Float32Field()  # API usage quota


class ClientInfo(models.DistributedModel):
    schema_version = 5

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "client_id", "purpose_tag")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # method call date and time (UNIX timestamp)
    client_id = fields.LowCardinalityField(fields.StringField())  # client identifier
    slots_total = fields.Int16Field()  # total amount of slots on the client
    slots_used = fields.Int16Field()  # amount of currently used slots on the client
    service_activity = fields.Enum8Field(fields.wrap_enum(ctc.ServiceActivity))  # (none/cleanup/reload/maintain)
    ncpu = fields.Int16Field()  # amount of CPU cores
    ram = fields.Int16Field()  # amount of RAM in Gb
    disk_total = fields.Int32Field()  # total amount of potentially available disk space for tasks in Gb
    disk_used = fields.Int32Field()  # amount of currently used disk space in Gb
    disk_reserved = fields.Int32Field()  # amount of reserved disk space in Gb
    disk_status = fields.Enum8Field(fields.wrap_enum(ctc.DiskStatus))  # disk status (ok/warning/critical)
    purpose_tag = fields.LowCardinalityField(fields.StringField())  # client purpose tag
    tags = fields.ArrayField(fields.StringField())  # all client tags
    alive = fields.Int8Field(default=0)  # whether the client is alive
    density_tags = fields.ArrayField(fields.StringField())  # all density client tags except MULTISLOT


class TaskOperation(models.DistributedModel):
    schema_version = 4

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "kind", "method", "task_type")
    )

    date = fields.DateField()  # operation date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # method call date and time (UNIX timestamp)
    kind = fields.Enum8Field(fields.wrap_enum(cts.OperationType))  # operation kind (svn/hg/rest/aapi/docker)
    method = fields.LowCardinalityField(fields.StringField())  # method (co/commit/... for svn, get/post/... for rest
    duration = fields.Int32Field()  # total duration of method calls in a minute or less, in milliseconds
    count = fields.Int32Field()  # method calls count in a minute or less
    task_id = fields.Int64Field()  # identifier of a task these calls were performed from
    task_type = fields.LowCardinalityField(fields.StringField())  # task type name
    host = fields.StringField()  # name (not a FQDN) of a host the calls took place on
    owner = fields.StringField()  # task owner, typically a group, but there may be exceptions


class ServiceQCounter(models.DistributedModel):
    schema_version = 4

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "name")
    )

    date = fields.DateField()  # counter date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # counter date and time (UNIX timestamp)
    name = fields.LowCardinalityField(fields.StringField())  # name of the counter
    count = fields.Int32Field()  # value of the counter
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which sent the counter


class ServiceQTask(models.DistributedModel):
    schema_version = 3

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "tag")
    )

    date = fields.DateField()  # counters date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # counters date and time (UNIX timestamp)
    tag = fields.LowCardinalityField(fields.StringField())  # client tag
    wants = fields.Int32Field()  # number of clients had wanted tasks
    got = fields.Int32Field()  # number of clients had got tasks
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which sent the counter


class ServiceQCall(models.DistributedModel):
    schema_version = 3

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "method")
    )

    date = fields.DateField()  # date of the call, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of the call (UNIX timestamp)
    method = fields.LowCardinalityField(fields.StringField())  # name of the method
    duration = fields.Int32Field()  # call processing duration, in milliseconds
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the call


class QuotaConsumption(models.DistributedModel):
    schema_version = 8

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "owner")
    )

    date = fields.DateField()  # date of the call, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of the call (UNIX timestamp)
    owner = fields.LowCardinalityField(fields.StringField())  # owner
    quota = fields.Int32Field()  # owner's quota
    pool = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # pool
    real_consumption = fields.Int32Field()  # quota consumption for last 12 hours
    future_consumption = fields.Int32Field()  # estimated quota consumption for next 12 hours
    executing_jobs = fields.Int32Field()  # number of currently executing jobs
    ram = fields.Int64Field()  # provided RAM, MiB
    cpu = fields.Int32Field()  # provided CPU, cores
    hdd = fields.Int64Field()  # provided space on HDD, MiB
    ssd = fields.Int64Field()  # provided space on SSD, MiB


class QuotaConsumptionDetails(models.DistributedModel):
    schema_version = 6

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # date of the call, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of the call (UNIX timestamp)
    task_id = fields.Int64Field()
    real_consumption = fields.Int32Field()  # quota consumption for last 12 hours
    ram = fields.Int64Field()  # provided RAM, MiB
    cpu = fields.Int32Field()  # provided CPU, cores
    hdd = fields.Int64Field()  # provided space on HDD, MiB
    ssd = fields.Int64Field()  # provided space on SSD, MiB
    pool = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # pool


class ServiceQSystemResources(models.DistributedModel):
    schema_version = 2

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "role")
    )

    date = fields.DateField()  # date of the call, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of the call (UNIX timestamp)
    user_cpu = fields.Int32Field()  # user CPU time usage, percents
    system_cpu = fields.Int32Field()  # system CPU time usage, percents
    rss = fields.Int32Field()  # resident memory size, Mb
    vms = fields.Int32Field()  # virtual memory size, Mb
    role = fields.LowCardinalityField(fields.StringField())  # role of the service instance
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the call


class ServiceQReplicationDelay(models.DistributedModel):
    schema_version = 2

    meta = dict(
        date_col="date",
        order_by=("date", "follower")
    )

    date = fields.DateField()  # counter date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # counter date and time (UNIX timestamp)
    delay = fields.Int32Field()  # value of the delay in number of operations
    follower = fields.LowCardinalityField(fields.StringField())  # FQDN of the follower for which delay is
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which sent the counter


class TaskDiskUsage(models.DistributedModel):
    schema_version = 4

    meta = dict(
        order_by=("date", "timestamp", "task_id", "task_type", "task_tags")
    )

    date = fields.DateField()  # event date (task execution end)
    timestamp = fields.DateTimeField()  # event timestamp (task execution end)

    max_working_set_usage = fields.Int64Field()  # maximal space consumed by a task in its working directory, bytes
    last_working_set_usage = fields.Int64Field()  # last known space consumed by a task in its working directory, bytes
    reserved_working_set = fields.Int64Field()  # disk space requirement, bytes
    resources_synced = fields.Int64Field()  # total size of resources synced into a working directory, bytes
    place_delta = fields.Int64Field()  # maximal space consumed by a task in total on /place partition, bytes

    task_id = fields.Int64Field()  # target task identifier
    task_type = fields.LowCardinalityField(fields.StringField())  # target task type
    owner = fields.LowCardinalityField(fields.StringField())  # target task owner
    task_tags = fields.ArrayField(fields.StringField())  # task tags, user-defined


class TemplateRunDelay(models.DistributedModel):
    schema_version = 1

    meta = dict(
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # delay record date
    timestamp = fields.DateTimeField()  # delay record time
    template_id = fields.Int32Field()  # task template identifier
    task_type = fields.StringField()  # task type
    delay = fields.Int32Field()  # delay value, in seconds
    owner = fields.StringField()  # template owner


class Audit(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "status", "task_id")
    )

    date = fields.DateField()  # audit record date
    timestamp = fields.DateTimeField()  # audit record timestamp
    status = fields.LowCardinalityField(fields.StringField())  # target status
    task_id = fields.Int64Field()  # task identifier
    duration = fields.Float32Field()  # time since previous status setup, seconds


class TaskCreation(models.DistributedModel):
    schema_version = 7

    meta = dict(
        order_by=("date", "timestamp", "owner", "task_type", "tags", "task_id")
    )

    date = fields.DateField()  # task creation date
    timestamp = fields.DateTimeField()  # task creation timestamp
    author = fields.LowCardinalityField(fields.StringField())  # task author
    owner = fields.LowCardinalityField(fields.StringField())  # task owner
    task_type = fields.LowCardinalityField(fields.StringField())  # task type
    unique_key = fields.NullableField(fields.StringField())  # deduplication (uniqueness) key
    hints = fields.ArrayField(  # task search hints (an array of arrays of length of two, [key, value])
        fields.StringField()
    )
    tags = fields.ArrayField(  # task tags
        fields.StringField()
    )
    task_id = fields.Int64Field()  # task identifier
    parent_id = fields.NullableField(fields.Int32Field())  # parent task identifier, if applicable

    #: scheduler identifier, if applicable. Positive numbers stand for automated runs, negative -- for manual ones
    scheduler_id = fields.NullableField(fields.Int32Field())
    sdk_type = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # Type of SDK


class TaskRejection(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "client_id", "reason", "owner", "task_id")
    )

    date = fields.DateField()  # task rejection date
    timestamp = fields.DateTimeField()  # task rejection timestamp
    task_id = fields.Int64Field()  # task identifier
    owner = fields.LowCardinalityField(fields.StringField())  # tasks owner
    client_id = fields.LowCardinalityField(fields.StringField())  # client id
    reason = fields.LowCardinalityField(fields.StringField())  # why the task was rejected


class QueueStatistics(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "owner", "task_type", "priority")
    )

    date = fields.DateField()  # signal date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # signal timestamp
    owner = fields.LowCardinalityField(fields.StringField())  # tasks owner
    task_type = fields.LowCardinalityField(fields.StringField())  # tasks type
    priority = fields.Int8Field()  # tasks priority
    amount = fields.Int32Field()  # tasks amount
    age = fields.Int64Field()  # tasks summary waiting time in queue (in seconds)
    wait_mutex_amount = fields.Int32Field()  # amount of tasks waiting semaphores
    wait_mutex_age = fields.Int64Field()  # waiting time of tasks in queue that waiting semaphores (in seconds)


class ExecutionStatistics(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "owner", "task_type", "status", "priority")
    )

    date = fields.DateField()  # signal date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # signal timestamp
    owner = fields.LowCardinalityField(fields.StringField())  # tasks owner
    task_type = fields.LowCardinalityField(fields.StringField())  # tasks type
    priority = fields.Int8Field()  # tasks priority
    amount = fields.Int32Field()  # tasks amount
    status = fields.LowCardinalityField(fields.StringField())  # tasks status


class Cleaner(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp"),
    )

    date = fields.DateField()  # date of Cleaner run
    timestamp = fields.DateTimeField()  # timestamp of Cleaner run

    removed_audits = fields.Int32Field()  # number of removed audit records
    removed_resources = fields.Int32Field()  # number of removed resource records
    removed_tasks = fields.Int32Field()  # number of removed task records
    tasks_switched_to_deleted = fields.Int32Field()  # number of tasks switched to DELETED state
    removed_notifications = fields.Int32Field()  # number of removed notification records
    removed_sessions = fields.Int32Field()  # number of removed sessions
    removed_parameters_metas = fields.Int32Field()  # number of removed metadata objects


class QuotaMedian(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "purpose_tag")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    purpose_tag = fields.LowCardinalityField(fields.StringField())  # client tag
    median = fields.Int32Field()  # median of remaining quota for last 100 owners for that purpose tag
    # median of (remaining quota / quota) for last 100 owners for the purpose tag
    ratio_median = fields.Int32Field(default=0)


class ResourceRegistration(models.DistributedModel):
    schema_version = 4

    meta = dict(
        order_by=("date", "timestamp", "owner", "resource_type", "purpose_tag")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    owner = fields.LowCardinalityField(fields.StringField())  # owner of registered resource
    resource_id = fields.Int64Field()  # id of registered resource
    resource_type = fields.LowCardinalityField(fields.StringField())  # type of registered resource
    size = fields.Int64Field()  # size of registered resource in bytes
    purpose_tag = fields.LowCardinalityField(fields.StringField())  # client tag
    client_id = fields.LowCardinalityField(fields.StringField())  # client id
    files_count = fields.Int32Field(default=0)


class ResourceSync(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    task_id = fields.Int64Field()  # id of task which synchronized the resource
    resource_id = fields.Int64Field()  # id of resource synchronized
    sync_duration = fields.Int32Field()  # in milliseconds
    client_id = fields.LowCardinalityField(fields.StringField())  # client id


class BackupResources(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    task_id = fields.Int64Field()  # backup task identifier
    resources_size = fields.Int64Field()  # size of backup resources in bytes
    resources_amount = fields.Int32Field()  # amount of backup resources
    dc = fields.LowCardinalityField(fields.StringField())  # name of data center
    backup_type = fields.LowCardinalityField(fields.StringField())  # backup or replication


class ResourcesOnStorages(models.DistributedModel):
    schema_version = 4

    meta = dict(
        order_by=("date", "timestamp", "owner", "type")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    owner = fields.LowCardinalityField(fields.StringField())  # resources' owner
    type = fields.LowCardinalityField(fields.StringField())  # resources' type
    amount = fields.Int32Field()  # amount of resources
    size = fields.Int64Field()  # size of resources in bytes
    disk_size = fields.Int64Field()  # size of resources in bytes

    mds_amount = fields.Int32Field()  # amount of resources
    mds_on_storages_amount = fields.Int32Field()  # amount of resources
    mds_size = fields.Int64Field()  # size of resources in bytes
    mds_on_storages_size = fields.Int64Field()  # size of resources in bytes

    amount_deleted = fields.Int32Field()  # amount of resources
    size_deleted = fields.Int64Field()  # size of resources in bytes
    disk_size_deleted = fields.Int64Field()  # size of resources in bytes
    mds_amount_deleted = fields.Int32Field()  # amount of resources
    mds_on_storages_amount_deleted = fields.Int32Field()  # amount of resources
    mds_size_deleted = fields.Int64Field()  # size of resources in bytes
    mds_on_storages_size_deleted = fields.Int64Field()  # size of resources in bytes

    mds_bucket_name = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # bucket mds name
    mds_bucket_used = fields.NullableField(fields.Int64Field())  # mds bucket used space
    mds_bucket_max_size = fields.NullableField(fields.Int64Field())  # mds bucket quota


class ResourcesSyncToMdsDelay(models.DistributedModel):
    schema_version = 1

    meta = dict(
        order_by=("date", "timestamp", "client_id")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    client_id = fields.LowCardinalityField(fields.StringField())
    amount = fields.Int32Field()  # amount of resources
    size = fields.Int64Field()  # size of resources in bytes


class ResourcesRemovedFromMds(models.DistributedModel):
    schema_version = 1

    meta = dict(
        order_by=("date", "timestamp", "bucket", "resource_type")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    bucket = fields.LowCardinalityField(fields.StringField())  # resource bucket
    delay = fields.Int64Field()  # delay of deleting in seconds
    state = fields.LowCardinalityField(fields.StringField())  # Deleted resource state
    resource_type = fields.LowCardinalityField(fields.StringField())  # resource type
    owner = fields.LowCardinalityField(fields.StringField())  # resource owner


class MdsBucketInfo(models.DistributedModel):
    schema_version = 1

    meta = dict(
        order_by=("date", "timestamp", "bucket")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    bucket = fields.LowCardinalityField(fields.StringField())  # bucket name
    meta_info_size = fields.Int64Field()  # size of meta information in bucket, bytes
    index_size = fields.Int64Field()  # size of index files for zipped TAR files, bytes
    objects_parts_size = fields.Int64Field()  # size of uncompleted uploads https://clubs.at.yandex-team.ru/storage/439
    deleted_objects_size = fields.Int64Field()  # size of removed from MDS, but not yet processed inside it


class TaskWaitDelay(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", )
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    owner = fields.LowCardinalityField(fields.StringField())
    task_id = fields.Int64Field()
    task_type = fields.LowCardinalityField(fields.StringField())
    status = fields.LowCardinalityField(fields.StringField())
    delay = fields.Int32Field()  # switch delay, seconds


class ExceptionStatistics(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "exc_type", "client_id", "component")
    )

    date = fields.DateField()  # date of signal
    timestamp = fields.DateTimeField()  # timestamp of signal
    exc_type = fields.LowCardinalityField(fields.StringField())  # type of exception
    client_tags = fields.ArrayField(fields.StringField())  # all client tags
    client_id = fields.LowCardinalityField(fields.StringField())  # client id
    component = fields.LowCardinalityField(fields.StringField())  # name of sandbox component where exception raised


class TaskExceptionStatistics(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()
    timestamp = fields.DateTimeField()
    exc_type = fields.LowCardinalityField(fields.StringField())
    client_tags = fields.ArrayField(fields.StringField())
    client_id = fields.LowCardinalityField(fields.StringField())
    task_id = fields.Int64Field()


class ApiQuotaConsumption(models.DistributedModel):
    schema_version = 3

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "login")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # method call date and time (UNIX timestamp)
    login = fields.LowCardinalityField(fields.StringField())  # user login
    consumption = fields.Int64Field()  # API usage consumption
    quota = fields.Int64Field()  # API usage quota
    source = fields.LowCardinalityField(fields.StringField())  # request source


class ServiceStatistics(models.DistributedModel):
    schema_version = 1

    meta = dict(
        date_col="date",
        order_by=("date", "timestamp", "service")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    service = fields.StringField()  # service (for SingletonService) or thread within a service (for ThreadedService)
    duration = fields.Int32Field()  # tick duration, in milliseconds


class TaskCpuUsage(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "start", "finish", "task_id")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    start = fields.DateTimeField()  # session start timestamp
    finish = fields.DateTimeField()  # session finish timestamp
    task_id = fields.Int64Field()  # task identifier

    p50 = fields.Int32Field()  # the 50th percentile
    p75 = fields.Int32Field()  # the 75th percentile
    p90 = fields.Int32Field()  # the 90th percentile


class TaskRamUsage(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "start", "finish", "task_id")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    start = fields.DateTimeField()  # session start timestamp
    finish = fields.DateTimeField()  # session finish timestamp
    task_id = fields.Int64Field()  # task identifier

    p50 = fields.Float32Field()  # the 50th percentile
    p75 = fields.Float32Field()  # the 75th percentile
    p90 = fields.Float32Field()  # the 90th percentile


class TaskDiskIoUsage(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    start = fields.DateTimeField()  # session start timestamp
    finish = fields.DateTimeField()  # session finish timestamp
    task_id = fields.Int64Field()  # task identifier

    p90 = fields.Float32Field()  # the 90th percentile
    total_bytes = fields.Int64Field()  # total number of read/written bytes


class TaskNetworkIoUsage(models.DistributedModel):
    schema_version = 2

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    start = fields.DateTimeField()  # session start timestamp
    finish = fields.DateTimeField()  # session finish timestamp
    task_id = fields.Int64Field()  # task identifier

    p90 = fields.Float32Field()  # the 90th percentile
    total_bytes = fields.Int64Field()  # total number of read/written bytes


class TaskboxCall(models.DistributedModel):
    schema_version = 4

    meta = dict(
        order_by=("date", "timestamp", "server")
    )

    date = fields.DateField()  # date of request start
    timestamp = fields.DateTimeField()  # date and time of request end
    method_name = fields.LowCardinalityField(fields.StringField())  # task's method name, for example, "on_create"
    server = fields.LowCardinalityField(fields.StringField())  # fqdn of the host which processed the request
    duration = fields.Int32Field()  # duration of the method execution, milliseconds
    author = fields.NullableField(fields.StringField())  # task author
    owner = fields.NullableField(fields.StringField())  # task owner
    resource_id = fields.NullableField(fields.Int64Field())  # tasks binary resource identifier
    result = fields.LowCardinalityField(fields.StringField())  # request result (success/error/...)
    age = fields.NullableField(fields.Int16Field())  # taskbox binary age


class TaskboxStatistics(models.DistributedModel):
    schema_version = 3

    meta = dict(
        order_by=("date", "timestamp", "server", "role")
    )

    date = fields.DateField()  # tick date
    timestamp = fields.DateTimeField()  # tick timestamp
    user_cpu = fields.Int32Field()  # user CPU time usage, percents
    system_cpu = fields.Int32Field()  # system CPU time usage, percents
    rss = fields.Int32Field()  # resident memory size, Mb
    vms = fields.Int32Field()  # virtual memory size, Mb
    role = fields.LowCardinalityField(fields.StringField())  # role of the service instance
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the call
    uuid = fields.NullableField(fields.StringField())  # UUID4 of the instance; None means dispatcher
    resource_id = fields.NullableField(fields.Int64Field())  # tasks binary resource identifier


class Notification(models.DistributedModel):
    schema_version = 4

    meta = dict(
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # tick date
    timestamp = fields.DateTimeField()  # tick timestamp
    author = fields.LowCardinalityField(fields.StringField())  # Notification author
    recipients = fields.ArrayField(fields.StringField())  # Notification recipients
    transport = fields.LowCardinalityField(fields.StringField())  # Notification transport
    sent = fields.Int8Field()  # Notification sent
    task_id = fields.NullableField(fields.Int64Field())  # Task id


class TaskHardwareMetrics(models.DistributedModel):
    schema_version = 5

    meta = dict(
        order_by=("date", "timestamp", "task_id")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    client_id = fields.LowCardinalityField(fields.StringField())  # client id
    task_id = fields.Int64Field()  # task identifier

    cpu = fields.Int16Field()  # in percentage

    ram = fields.Int64Field()  # in bytes

    disk_read = fields.Int64Field()  # in bytes
    disk_write = fields.Int64Field()  # in bytes

    disk_read_ops = fields.Int64Field()  # number of operations
    disk_write_ops = fields.Int64Field()  # number of operations

    net_read = fields.Int64Field()  # in bytes
    net_write = fields.Int64Field()  # in bytes


class ProxyRequestsDynamicNets(models.DistributedModel):
    schema_version = 3
    meta = dict(
        order_by=("date", "timestamp", "resource_id", "user")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # request date and time (UNIX timestamp)
    resource_id = fields.Int64Field()  # resource id
    resource_type = fields.LowCardinalityField(fields.StringField())  # resource type
    remote_ip = fields.StringField()  # remote ip
    user_agent = fields.NullableField(fields.StringField())  # user agent of request sender
    user = fields.LowCardinalityField(fields.StringField())  # user name
    net_macros = fields.LowCardinalityField(fields.StringField())  # net macros
    host = fields.StringField()  # host header
    referer = fields.StringField()  # referer header


class ProxyApiCall(models.DistributedModel):
    schema_version = 2
    meta = dict(
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of request (UNIX timestamp)
    user = fields.StringField()  # login of the request author
    remote_ip = fields.StringField()  # remote ip

    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the request
    request_id = fields.StringField()  # request identifier
    endpoint = fields.StringField()  # endpoint template; may contain spaces
    path = fields.StringField()  # url path without query string
    query_string = fields.StringField()  # a string of "key=value&another_key=another_value" format; keys are sorted

    data_source = fields.LowCardinalityField(fields.StringField())  # source of data: S3 bucket or FQDN of the host
    data_size = fields.Int64Field()  # size of transferred data in bytes (Content-Length)
    resource_id = fields.Int64Field()  # resource id
    resource_type = fields.LowCardinalityField(fields.StringField())  # resource type
    resource_owner = fields.LowCardinalityField(fields.StringField())  # resource owner

    response_code = fields.Int16Field()  # HTTP response code

    # Request execution metrics, durations are in milliseconds
    total_duration = fields.Int32Field()  # total request duration
    api_duration = fields.Int32Field(default=0)  # duration of sub-requests to API


class MdsApiCall(models.DistributedModel):
    schema_version = 1
    meta = dict(
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # request date, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of request (UNIX timestamp)
    user = fields.StringField()  # login of the original request author
    server = fields.LowCardinalityField(fields.StringField())  # FQDN of the host which processed the request

    proxy_request_id = fields.StringField()  # proxy request identifier
    mds_request_id = fields.StringField()  # request identifier
    method = fields.LowCardinalityField(fields.StringField())  # HTTP method name
    path = fields.StringField()  # url path without query string
    response_code = fields.Int16Field()  # HTTP response code

    bucket_id = fields.LowCardinalityField(fields.StringField())  # S3 bucket
    resource_id = fields.Int64Field()  # resource id
    data_size = fields.Int64Field()  # size of transferred data in bytes (Content-Length)

    total_duration = fields.Int32Field()  # total request duration in milliseconds


class EmptyTaskletMetrics(models.DistributedModel):
    schema_version = 1
    meta = dict(
        order_by=("date", "timestamp", "run_type", "measure_type")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)
    run_type = fields.LowCardinalityField(fields.StringField())  # type of launched tasklet
    measure_type = fields.LowCardinalityField(fields.StringField())  # type of collected metrics
    request_type = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # type of request
    profiler_type = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # source of profile
    name = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # metric name
    value = fields.Int64Field()  # time in microseconds


class TaskIntervals(models.DistributedModel):
    schema_version = 1
    meta = dict(
        order_by=("date", "timestamp", "task_type", "task_owner", "task_id", "pool")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)

    task_id = fields.Int64Field()  # task identifier
    task_owner = fields.LowCardinalityField(fields.StringField())  # task owner
    task_type = fields.LowCardinalityField(fields.StringField())  # task type name

    consumption = fields.Int64Field()  # QP consumption of task in interval
    duration = fields.Int64Field()  # interval duration
    start = fields.DateTimeField()  # time interval started
    pool = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # quota pool

    client_id = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # execution host
    client_tags = fields.ArrayField(fields.LowCardinalityField(fields.StringField()))  # execution client tags

    privileged = fields.Int32Field()  # task privileged flag
    cores = fields.NullableField(fields.Int32Field())  # task required cores
    ram = fields.NullableField(fields.Int64Field())  # task required ram
    caches = fields.Int32Field()  # task declared caches
    disk = fields.NullableField(fields.Int64Field())  # task required disk


class Coredumps(models.DistributedModel):
    schema_version = 1
    meta = dict(
        order_by=("date", "timestamp")
    )

    date = fields.DateField()  # date of service tick, %Y-%m-%d format
    timestamp = fields.DateTimeField()  # date and time of service tick (UNIX timestamp)

    path = fields.StringField()  # coredump path
    pid = fields.NullableField(fields.Int64Field())  # process pid
    rpid = fields.NullableField(fields.Int64Field())  # process real pid on host system
    core_limit = fields.NullableField(fields.Int64Field())  # coredump file size limit

    client_id = fields.LowCardinalityField(fields.NullableField(fields.StringField()))  # coredump host
    client_tags = fields.ArrayField(fields.LowCardinalityField(fields.StringField()))  # coredump client tags

    task_id = fields.NullableField(fields.Int64Field())  # task identifier


SIGNAL_MODELS = {
    class_.__name__.lower(): class_
    for class_ in filter(
        lambda c: inspect.isclass(c) and issubclass(c, models.DistributedModel),
        locals().values()
    )
}
