import abc
import inspect

import sys
import six

from sandbox.common import enum
from sandbox.common import patterns
import sandbox.common.types.statistics as ctss


class Proxy(enum.Enum):
    enum.Enum.lower_case()

    ARNOLD = None
    HAHN = None


def is_dunder(s):
    return len(s) >= 4 and s[:2] == s[-2:] == "__"


@six.add_metaclass(abc.ABCMeta)
class _Column(patterns.Abstract):
    __counter__ = 0

    @abc.abstractmethod
    def __cast__(_, value):
        pass

    type = None

    __slots__ = ("__no__", "required", "sort_order")
    __defs__ = (0, True, None)

    def __init__(self, *args, **kwargs):
        super(_Column, self).__init__(*args, **kwargs)
        self.__no__ = _Column.__counter__
        _Column.__counter__ += 1

    def __iter__(self):
        yield "type", self.type
        for k, v in super(_Column, self).__iter__():
            if not is_dunder(k) and v is not None:
                yield k, v

    def itervalues(self):
        for _, v in self:
            yield v

    @classmethod
    def cast(cls, value):
        try:
            return cls.__cast__(value)
        except Exception as exc:
            raise ValueError(exc)


class StringColumn(_Column):
    type = "string"
    __cast__ = str


class Int32Column(_Column):
    type = "int32"
    __cast__ = int


class Int64Column(_Column):
    type = "int64"
    __cast__ = int


class YTSignalMetadata(patterns.Abstract):
    __slots__ = ("signal_type", "proxy", "path")
    __defs__ = (None,) * 3


class _YTSignalMeta(type):
    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            return super(_YTSignalMeta, mcs).__new__(mcs, name, bases, namespace)

        missing_fields = [k for k, v in namespace.get("meta", YTSignalMetadata()) if v is None]
        if missing_fields:
            raise TypeError("Missing table metadata values: {}".format(", ".join(missing_fields)))

        schema = []
        required, known = set(), set()
        for name, member in sorted(
            ((k, v) for k, v in namespace.items() if isinstance(v, _Column)),
            key=lambda item: item[1].__no__
        ):
            field_specs = dict(member)
            field_specs.update(name=name)
            if member.required:
                required.add(name)
            known.add(name)
            schema.append(field_specs)

        namespace["schema"] = schema
        namespace["__required_fields__"] = required
        namespace["__known_fields__"] = known
        return super(_YTSignalMeta, mcs).__new__(mcs, name, bases, namespace)


@six.add_metaclass(_YTSignalMeta)
class YTSignal(object):
    @classmethod
    def make(cls, **kwargs):
        field_names = set(kwargs.keys())
        missing_required = cls.__required_fields__ - field_names
        if missing_required:
            raise ValueError("Missing fields: {}".format(", ".join(missing_required)))
        unknown = field_names - cls.__known_fields__
        if unknown:
            raise ValueError("Unknown fields: {}".format(", ".join(unknown)))

        res = dict(type=cls.meta.signal_type)
        for key, value in kwargs.items():
            column = getattr(cls, key)
            try:
                res[key] = column.cast(value)
            except ValueError:
                _, _, tb = sys.exc_info()
                six.reraise(
                    ValueError, ValueError("Failed to convert {!r}={!r} to {!r}".format(key, value, column.type)), tb
                )

        return res


class APICallYT(YTSignal):
    # Deprecated, not used
    meta = YTSignalMetadata(
        signal_type=ctss.YTSignalType.API_CALL_YT,
        proxy=Proxy.HAHN,
        path="//home/sandbox/{environment}/logs/api/access"
    )

    timestamp = StringColumn()
    server = StringColumn()
    id = StringColumn()
    ip = StringColumn()
    user = StringColumn()
    owner = StringColumn()
    rtype = StringColumn()
    path = StringColumn()
    query_string = StringColumn()
    code = Int32Column()
    rlength = Int32Column()
    ua = StringColumn()
    rtime = Int32Column()


class ResourceAudit(YTSignal):
    meta = YTSignalMetadata(
        signal_type=ctss.YTSignalType.RESOURCE_AUDIT,
        proxy=Proxy.HAHN,
        path="//home/sandbox/{environment}/logs/core/resources_audit"
    )

    timestamp = StringColumn()
    id = Int64Column()
    state = StringColumn()
    attributes = StringColumn()
    author = StringColumn()
    message = StringColumn()


YT_MODELS = {
    class_.meta.signal_type: class_
    for class_ in filter(
        lambda c: inspect.isclass(c) and issubclass(c, YTSignal) and c is not YTSignal,
        locals().values()
    )
}
