import logging
import datetime as dt

import sandbox.common.types.task as ctt

from sandbox.services import base
from sandbox.yasandbox import controller


logger = logging.getLogger(__name__)


class TasksEnqueuer(base.Service):
    """
    System thread which switches tasks from status ENQUEUING to one of the follows: ENQUEUED, WAIT_RES or EXCEPTION
    """

    tick_interval = 5
    NOTIFICATION_TIMEOUT = 7

    def tick(self):
        tick_finish = dt.datetime.utcnow() + dt.timedelta(minutes=self.NOTIFICATION_TIMEOUT - 1)
        cache = controller.TaskQueue.HostsCache()

        while not self._stop_requested.is_set() and dt.datetime.utcnow() < tick_finish:
            task_id = controller.TaskQueue.qclient.prequeue_pop()
            if task_id is None:
                # No more tasks to enqueue
                break

            logger.debug("Take task #%s", task_id)
            task = controller.Task.get(task_id)
            if task.execution.status != ctt.Status.ENQUEUING:
                logger.warning("Task #%s not in status %s, skipping", task_id, ctt.Status.ENQUEUING)
                continue

            result = controller.TaskQueue.finalize_enqueue_task(
                task=task, cache=cache, logger=logger, reset_lock=True
            )

            if result == controller.TaskQueue.EnqueueResult.NEED_VALIDATION:
                break
