import re
import time
import logging
import threading
import datetime as dt
import operator as op
import collections

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from sandbox.deploy import juggler

from .client_availability_manager import ClientAvailabilityManager

from sandbox.services.base import service as base_service
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


logger = logging.getLogger(__name__)

MailInfo = collections.namedtuple("MailInfo", ("mail", "uid"))
MailThread = collections.namedtuple("MailThread", ("ids", "messages"))
DutyInfo = collections.namedtuple(
    "DutyInfo",
    (
        "assigned_duty_login",
        "dead_clients_ids",
        "next_duties"
    )
)


def prettify_time(total):
    """
    Make time duration more readable

    :param total: time duration in minutes
    :return: a pretty string (for example, "5 days 1 hour 17 minutes")
    """

    if not total:
        return "now"

    suffixes = ("minute", "hour", "day")
    divisors = (60, 24, 365)
    res = []
    for k in range(len(divisors)):
        quantity = total % divisors[k]
        if quantity:
            res += [suffixes[k] + ("s" if quantity > 1 else ""), str(quantity)]
        total /= divisors[k]
    return " ".join(reversed(res))


class UnrepliedMailInfo(collections.namedtuple("UnrepliedMailInfo", ("from_who", "subject", "delay", "uid"))):
    """
    A structure representing a single mail thread (__repr__ returns an HTML-formatted preview)
    """

    def __repr__(self):
        delay = prettify_time(self.delay)
        if self.delay:
            delay += " ago"
        return "<code>#{uid}</code> | <b>{subject}</b> | {from_who} | {delay}".format(
            from_who=self.from_who,
            subject=self.subject.replace("<", "&lt;").replace(">", "&gt;"),
            uid=self.uid,
            delay=delay
        )


class TelegramBot(base_service.SingletonService):
    """
    Bot to look after tasks code autodeploy state.
    It sends notifications to a chat specified in settings and highlights the current duty if possible. Topics:
    - Emergency chat unreplied messages
    - Tasks code autodeploy breakage for more than half an hour.

    You can make the bot to ignore certain mails (not threads) by sending it a
    "ignore (a list of mail identifiers, NOT Message-Ids, goes here)" message in public or privately.
    """

    tick_interval = 10

    IMAP_HOST = "imap.yandex-team.ru"  # host to fetch mailing list from
    SOCKET_TIMEOUT = 30  # for IMAP operations, seconds

    # projects which have DNS automation disabled
    IGNORE_DNS_AUTOMATION = (
        "mac-iostest-sandbox",
        "mobile-mac-sandbox",
        "sandbox-macs",
    )

    # projects which have healing automation disabled
    IGNORE_HEALING_AUTOMATION = (
        "mac-iostest-sandbox",
    )

    SEARCH_CRITERIONS = "(SINCE {})"  # mail search criterions
    MAIL_LOOKUP_THRESHOLD = 14  # mail thread depth in days (don't load mails, which are older than this, at all)

    # parts to fetch for every mail; more headers on https://tools.ietf.org/html/rfc822#section-4.6
    FETCH_MESSAGE_PARTS = "(BODY[HEADER.FIELDS (Date Subject Message-Id From In-Reply-To References)])"

    LOGIN_RE = re.compile(r"(?P<login>[-\w_.+]+)@[-\w.]+\.[-\w]+")  # sender login regexp
    MESSAGE_LENGTH_LIMIT = 4096  # max. Telegram message length; https://core.telegram.org/method/messages.sendMessage
    MESSAGES_IN_CONTEXT_LIMIT = 15  # keep last N sent Telegram messages

    SANDBOX_STAFF_GROUP = "svc_sandbox_development"  # Staff group to fetch Sandbox members from
    STAFF_TTL = 300  # how often to fetch the aforementioned group's contents, seconds

    DUTY_LIMIT = 1

    # assume that Sandbox team reads all messages in sandbox-emergency@ left in this amount of seconds after theirs;
    # helps avoiding false positives coming from "Thanks!", "Alright" and such
    ATTENTION_SPAN_WINDOW = 2 * 60

    # response text for task tracking request in wrong format
    WRONG_TASK_TRACKING_REQUEST_REPLY = "Expected request format: /track <task_id> [<status> <status> ...]"

    # Set of Telegram logins, whose messages do not trigger emergency channel check
    EMERGENCY_BLACKLIST = {"YaIncBot"}

    # Max Telegram user absence interval
    MAX_ABSENCE_INTERVAL = dt.timedelta(days=90)

    SMILE = "\xF0\x9F\x98\x91"  # utf-8 :| smile code

    @property
    def bot_commands(self):
        return {
            "/dead_clients": self.get_dead_clients,
            "/status": self.get_status,
        }

    def _bot_owner_last_visit(self):
        last_visit_str = self.context.get(self.ContextField.BOT_OWNER_LAST_VISIT)
        if not last_visit_str:
            return "[warning] Bot owner has never written any message to the bot.\n"
        last_visit = common.format.str2dt(last_visit_str)
        msg = "Bot owner's last visit: {}\n".format(common.format.utcdt2iso(last_visit))
        if dt.datetime.utcnow() - last_visit > self.MAX_ABSENCE_INTERVAL:
            msg += "[warning] Last visit was too long time ago. " \
                   "Send any message to the bot using bot owner account to ensure it is alive.\n"
        return msg

    def get_automation_info(self):
        msg = "Automation on projects:\n"
        statuses = self.context.setdefault(self.ContextField.WALLE_AUTOMATION_STATUSES, {})
        for project, status in statuses.iteritems():
            msg += "Project '{}': {}\n".format(project, "enabled" if status else "disabled")
        return msg

    def get_status(self):
        statuses = [
            self._bot_owner_last_visit(),
            self.get_dead_clients(),
            self.get_automation_info()
        ]
        return "\n".join(filter(None, statuses))

    def get_dead_clients(self):
        return "Dead clients:\n{}\n".format(", ".join(self.duty.dead_clients_ids))

    class Command(common.utils.Enum):
        """ IMAP commands """
        SEARCH = None
        FETCH = None

    class ContextField(common.utils.Enum):
        """ This service process' context field constants """
        common.utils.Enum.lower_case()

        OFFSET = None
        TELEGRAM_IDS = None
        IGNORED_UNREPLIED = None
        IGNORED_FRESH = None
        FRESH_MAIL_NOTIFICATIONS = None
        UNREPLIED_MAIL_NOTIFICATIONS = None
        AUTODEPLOY_NOTIFICATIONS = None
        EMERGENCY_CHAT_MESSAGES = None
        EMERGENCY_NOTIFICATIONS = None
        EMERGENCY_CHECK_STATUS = None
        USERNAMES = None
        UNREPLIED = None
        FRESH = None
        BOT_OWNER_LAST_VISIT = None
        WALLE_AUTOMATION_STATUSES = None
        WALLE_AUTOMATION_NOTIFICATION = None

    def __init__(self):
        super(TelegramBot, self).__init__()

        self.duty = None
        self.workers = {}
        self.is_working = True

    @property
    def worker_settings(self):
        return {
            func.__name__: func
            for func in (
                self.messages_handler,
            )
        }

    def messages_handler(self):
        while self.is_working:
            self.context[self.ContextField.OFFSET] = self.process_updates(
                self.context.get(self.ContextField.OFFSET)
            )

            time.sleep(1)

    def process_updates(self, offset):
        messages = self.telegram.get_updates(offset=offset).get("result", [])

        if messages:
            for msg in messages:
                try:
                    self.__handle_message(msg)
                except Exception as ex:
                    logger.exception("Caught exception: {}, ignoring message {}".format(ex, msg))

            return messages[-1]["update_id"] + 1

        return offset

    def __make_worker(self, func):
        def wrapper():
            try:
                func()
            except Exception:
                logger.exception("%s: caught exception", func.__name__)

        worker = threading.Thread(
            target=wrapper,
            name=func.__name__,
        )
        return worker

    @staticmethod
    def __tasks(query, limit):
        return list(mapping.Task.objects(**query).order_by("-id").limit(limit))

    def fetch_duty(self):
        """
        Get today's duty (based on configuration parameters)
        """
        # duty detection tasks query and limit
        duty_query = {
            "execution__status": ctt.Status.SUCCESS,
            "scheduler__in": [self.service_config["duty_scheduler"], -self.service_config["duty_scheduler"]],
        }

        logger.debug("Fetching duty login")
        task = self.__tasks(duty_query, self.DUTY_LIMIT)[0]

        params = {
            param.key: param.value for param in task.parameters.output
        }

        duty_state = DutyInfo(
            assigned_duty_login=params.get("assigned_duty_login", "").lower(),
            dead_clients_ids=params.get("dead_clients_ids", []),
            next_duties=params.get("next_duties")
        )

        return duty_state

    @common.utils.singleton_property
    def staff_token(self):
        return common.utils.read_settings_value_from_file(common.config.Registry().server.auth.oauth.token)

    @property
    @common.utils.ttl_cache(STAFF_TTL)
    def sandbox_team(self):
        """
        Fetch Staff and Telegram logins of Sandbox team once in a `TelegramBot.STAFF_TTL` seconds

        :return: Staff login -> Telegram login dictionary
        (may contain ``None`` if a user doesn't have an account)
        :rtype: dict
        """

        logger.debug("Fetching Sandbox team contents from Staff")
        staff = common.rest.ThreadLocalCachableClient(
            base_url=common.config.Registry().server.auth.staff.url,
            auth=self.staff_token,
            logger=logger,
            ua="Sandbox.TelegramBot"
        )

        members = staff.groupmembership.read(
            params={
                "group.url": self.service_config["staff_group"],
                "_fields": ",".join(("person.login", "person.official.is_dismissed")),
            }
        ).get("result", [])

        staff_logins = [
            member["person"]["login"]
            for member in members if
            not member["person"]["official"]["is_dismissed"]
        ]
        persons = staff.persons.read(
            params={
                "login": ",".join(staff_logins),
                "_fields": ",".join(("login", "accounts")),
            },
        ).get("result", [])

        telegram_logins = {}
        for person in persons:
            lazy_values = iter(account for account in person["accounts"] if account["type"] == "telegram")
            telegram = next(lazy_values, {}).get("value", "").lower()
            telegram_logins[person["login"]] = telegram

        if not telegram_logins:
            raise ValueError("Staff says that there's no one in Sandbox team, probably their API is down")
        return telegram_logins

    @common.utils.singleton_property
    def telegram(self):
        """
        :return: Telegram bot instance
        :rtype: common.telegram.TelegramBot
        """

        null_logger = logging.getLogger("telegram")
        null_logger.handlers = [logging.NullHandler()]
        null_logger.propagate = False
        return common.telegram.TelegramBot(
            common.utils.read_settings_value_from_file(self.service_config["telegram"]["token"]),
            null_logger
        )

    def notify(
        self, message, key, personal=False,
        ignore_interval=False, ignore_work_life_balance=False,
        delete_previous=True
    ):
        """
        Send a message to a chat specified in settings, highlight the duty if possible.
        If we're outside the working hours window
        (or a message has already been sent within a certain interval), do nothing.

        :param message: chat message contents (simple HTML allowed)
        :param key: context key for reading/recording notification time into
        :param personal: send a private message if a user's Telegram ID is available
        :param ignore_interval: ignore last notification time;
          if False, only send messages once in an `interval` minutes
        :param ignore_work_life_balance: send notifications outside of working hours
        :param delete_previous: delete previous notification from the chat
        :return: whether the notification was sent
        """

        notifications = self.service_config["telegram"]["notifications"]
        day_start = dt.timedelta(hours=notifications["day_start"])
        day_end = dt.timedelta(hours=notifications["day_end"])
        interval = notifications["interval"]

        local_now = dt.datetime.now()
        utcnow = dt.datetime.utcnow()
        sent_notifications = self.context.get(key, [])
        last_notification_time = (
            sent_notifications[-1]["time"]
            if sent_notifications else
            utcnow - dt.timedelta(minutes=interval)
        )

        today = dt.datetime(*local_now.timetuple()[:3])
        appropriate_time = today + day_start <= local_now <= today + day_end
        too_often = (utcnow - last_notification_time).total_seconds() / 60 < interval

        sent = False
        going_to_send = (
            self.duty and
            (appropriate_time or ignore_work_life_balance) and
            any((personal, not too_often, ignore_interval))
        )
        if not going_to_send:
            reasons = []
            if not (appropriate_time or ignore_work_life_balance):
                reasons.append("shh, no notifications, only dreams")
            if not self.duty:
                reasons.append("no duty (failed to fetch them from Sandbox task?)")
            if too_often:
                reasons.append("attempting more than once in {} minutes".format(interval))
            logger.info("No message is sent (%s): %s", key, "; ".join(reasons))
            return False

        duty_telegram = self.sandbox_team.get(self.duty.assigned_duty_login)
        if duty_telegram:
            message = ", ".join(("@{}".format(duty_telegram), message))

        chat_id = self.service_config["telegram"]["chat"]
        if personal:
            personal_id = self.context.get(self.ContextField.TELEGRAM_IDS, {}).get(duty_telegram)
            if not personal_id:
                logger.info("%s hasn't ever written to me, I don't know their Telegram ID")
            else:
                chat_id = personal_id

        logger.debug(
            "Sending a message to %d (personal: %s)",
            chat_id, chat_id != self.service_config["telegram"]["chat"]
        )
        result = self.telegram.send_message(
            chat_id, common.utils.force_unicode_safe(message), common.telegram.ParseMode.HTML
        )

        if not result["ok"]:
            logger.error("Failed to send notification (%s): %s", key, result["description"])
        else:
            sent = True
            if sent_notifications and delete_previous:
                ids = map(sent_notifications[-1].get, ("chat_id", "message_id"))
                if None not in ids:
                    self.telegram.delete_message(*ids)

            sent_notifications.append(dict(
                chat_id=chat_id,
                message_id=result["result"]["message_id"],
                time=utcnow,
            ))
            self.context[key] = sent_notifications[-self.MESSAGES_IN_CONTEXT_LIMIT:]

        return sent

    @common.utils.singleton_property
    def bot_id(self):
        return self.telegram.get_me()["result"]["id"]

    def send_message(self, source, reply, message, username):
        result = self.telegram.send_message(
            chat_id=source,
            text=common.utils.force_unicode_safe(reply),
            reply_to_message_id=message["message_id"],
            disable_notification=True,
            parse_mode=common.telegram.ParseMode.HTML,
        )
        if not result["ok"]:
            logger.error(
                "Failed to reply to message #%s by %s: %s",
                message["message_id"], username, result["description"]
            )

    def track_task(self, message, source, username):
        splited_message = message["text"].lower().strip().split()

        if len(splited_message) < 2:
            return self.WRONG_TASK_TRACKING_REQUEST_REPLY

        task_id, statuses = splited_message[1], splited_message[2:]

        try:
            task_id = int(task_id.strip("#"))
        except (TypeError, ValueError):
            return self.WRONG_TASK_TRACKING_REQUEST_REPLY

        task = mapping.Task.objects.with_id(task_id)

        if task is None:
            return "Task {} doesn't exist".format(task_id)

        statuses = [status.strip(',').upper() for status in statuses]
        if not statuses:
            statuses = [
                ctt.Status.SUCCESS, ctt.Status.FAILURE,
                ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                ctt.Status.TIMEOUT, ctt.Status.EXPIRED
            ]

        try:
            notification = controller.Notification.notification(
                ctn.Transport.TELEGRAM,
                statuses,
                [username]
            )

            for task_notification in task.notifications:
                if all(
                    getattr(task_notification, field) == getattr(notification, field)
                    for field in ("transport", "statuses", "recipients")
                ):
                    return "The same notification is already added"

            task.notifications.append(notification)
            mapping.Task.objects(id=task_id).update_one(set__notifications=task.notifications)
            controller.TaskStatusNotifierTrigger.append(
                task_id, notification,
                resolver=lambda *x: x
            )
            return "Tracking added"
        except Exception as ex:
            logger.error("Error in appending notification trigger.", exc_info=ex)
            return "Sorry, it was strange database error, try again later or ask sandbox team to help"

    def handle_sandbox_message(self, message, sender_id, username):
        initial_message_id = message.get("reply_to_message", {}).get("message_id")
        self.context.setdefault(self.ContextField.TELEGRAM_IDS, {})[username] = sender_id
        message_text = message.get("text", None)
        logger.debug("Message from Sandbox team (@%s): %s", username, message_text)
        if not message_text:
            return

        reply = self.SMILE
        message_text = message_text.lower().strip()
        if any(message_text.startswith(word) for word in ("ignore", "/ignore")):
            if initial_message_id in map(
                op.itemgetter("message_id"),
                self.context.get(self.ContextField.EMERGENCY_NOTIFICATIONS, [])
            ):
                messages = self.context.get(self.ContextField.EMERGENCY_CHAT_MESSAGES, [])
                while messages and messages[-1]["from"] not in self.sandbox_team.itervalues():
                    messages.pop()
                self.context[self.ContextField.EMERGENCY_CHAT_MESSAGES] = messages
                juggler.EmergencyChannel.ok("Silenced by @{}".format(username))
                reply = "Juggler check is set to OK"
        return reply

    def __handle_message(self, full_message):
        if "my_chat_member" in full_message:  # bot is [un]blocked by user
            return
        message = full_message["message"]
        if any(
            key in message
            for key in
            (
                "new_chat_participant",
                "new_chat_member",
                "new_chat_members",
                "left_chat_participant",
                "left_chat_member",
                "left_chat_members",
            )
        ):
            return

        username = message["from"].get("username")
        if not (username and controller.user.User.check_telegram_username(username)):
            return

        username = username.lower()
        user_cache = self.context.get(self.ContextField.USERNAMES, {})
        source = message["chat"]["id"]
        sender_id = message["from"]["id"]
        user_cache[username] = sender_id

        initial_message_author = message.get("reply_to_message", {}).get("from", {}).get("id")

        if "emergency" in self.service_config and source == self.service_config["emergency"]["chat"]:
            logger.debug("Recording message from @%s in sandbox-emergency@", username)
            self.context.setdefault(self.ContextField.EMERGENCY_CHAT_MESSAGES, []).append({
                "date": message["date"],
                "from": username,
            })

        if sender_id == common.config.Registry().server.services.telegram_bot.telegram.owner_id:
            self.context[self.ContextField.BOT_OWNER_LAST_VISIT] = common.format.dt2str(dt.datetime.utcnow())

        reply = self.SMILE

        personal = any((
            initial_message_author == self.bot_id,
            sender_id == source
        ))
        if not personal:
            return

        message_text = message.get("text", None)
        if not message_text:
            return
        message_text = message_text.lower().strip()
        if message_text.startswith("/track"):
            reply = self.track_task(message, source, username)
        elif message_text in self.bot_commands:
            try:
                reply = self.bot_commands[message_text]()
            except Exception as ex:
                logger.exception("Caught exception {}".format(ex))
        elif username in self.sandbox_team.itervalues():
            reply = self.handle_sandbox_message(message, sender_id, username)

        if reply == self.SMILE and message.get("reply_to_message") and username not in self.sandbox_team.itervalues():
            # make Kate silent for replies on her own messages (see DMP-56 for reference)
            return

        self.send_message(source, reply, message, username)

    def check_emergency(self):
        logger.debug("Checking sandbox-emergency@ status")
        utcnow = int(time.time())
        messages = self.context.get(self.ContextField.EMERGENCY_CHAT_MESSAGES, [])
        settings = self.service_config["emergency"]

        delta = 0
        blacklist = set(self.sandbox_team.values()) | self.EMERGENCY_BLACKLIST
        if messages and messages[-1]["from"] not in blacklist:
            for message in reversed(messages):
                if (
                    message["from"] in self.sandbox_team.itervalues() and
                    (messages[-1]["date"] - message["date"]) < self.ATTENTION_SPAN_WINDOW
                ):
                    break
            else:
                delta = utcnow - messages[-1]["date"]
        delay = delta / 60, delta % 60

        status = (
            ctm.JugglerCheckStatus.OK
            if delay[0] < settings["warning_timeout"] else
            ctm.JugglerCheckStatus.WARNING
            if delay[0] < settings["critical_timeout"] else
            ctm.JugglerCheckStatus.CRITICAL
        )
        message = (
            "all good"
            if status == ctm.JugglerCheckStatus.OK else
            "activity in sandbox-emergency@ ({}m {}s ago)".format(*delay)
        )
        logger.debug("%s (%s); last message: %r", message, status, messages[-1] if messages else None)

        prev_status = self.context.get(self.ContextField.EMERGENCY_CHECK_STATUS)
        self.context[self.ContextField.EMERGENCY_CHECK_STATUS] = status

        juggler_reply = getattr(juggler.EmergencyChannel, status.lower())(message)
        logger.debug("Reply from Juggler: %r", juggler_reply)
        if status != ctm.JugglerCheckStatus.OK and status != prev_status:
            self.notify(
                "there's {} ({})".format(message, status),
                key=self.ContextField.EMERGENCY_NOTIFICATIONS,
                ignore_interval=True, ignore_work_life_balance=True
            )
        self.context[self.ContextField.EMERGENCY_CHAT_MESSAGES] = messages[-self.MESSAGES_IN_CONTEXT_LIMIT:]

    def __notify(self, name, projects):
        if not len(projects):
            return
        self.notify(
            "{} is disabled or in unknown status for project(s) {}".format(
                name,
                ", ".join(projects),
            ),
            key=self.ContextField.WALLE_AUTOMATION_NOTIFICATION,
        )

    def check_automation_statuses(self):
        obj = mapping.Service.objects.with_id(ClientAvailabilityManager.name)
        if obj is None:
            return

        checker_context = obj.context.get(ClientAvailabilityManager.ContextField.AUTOMATION, {})

        dns_check_disabled_projects = set()
        healing_check_disabled_projects = set()
        for project, status in checker_context.iteritems():
            if not status.get("dns_automation_enabled", False) and project not in self.IGNORE_DNS_AUTOMATION:
                dns_check_disabled_projects.add(project)
            if not status.get("healing_automation_enabled", False) and project not in self.IGNORE_HEALING_AUTOMATION:
                healing_check_disabled_projects.add(project)
        self.__notify("DNS-automation", dns_check_disabled_projects)
        self.__notify("Healing-automation", healing_check_disabled_projects)

    def tick(self):
        if not self.context.get(self.ContextField.USERNAMES):
            self.context[self.ContextField.USERNAMES] = {}

        self.duty = self.fetch_duty()

        try:
            if "emergency" in self.service_config:
                self.check_emergency()
        except (common.telegram.TelegramError, common.rest.Client.HTTPError) as exc:
            logger.exception("Failed to check sandbox-emergency@ status: %s", exc)

        try:
            self.check_automation_statuses()
        except Exception as exc:
            logger.exception("Failed to check automation status: %s", exc)

        if not self.workers:
            for func_name, func in self.worker_settings.items():
                worker = self.__make_worker(func)
                self.workers[func_name] = worker
                worker.start()

        for key in self.workers:
            worker = self.workers[key]
            if worker.is_alive():
                continue
            logger.warning("%s: respawning the dead thread", key)
            self.workers[key] = self.__make_worker(*self.worker_settings[key])

    def on_stop(self):
        self.is_working = False

        for name, worker in self.workers.iteritems():
            logger.info("%s: waiting for the stop", name)
            worker.join()
