from __future__ import unicode_literals

import logging

import six

from sandbox.services import base
from sandbox.common import context
from sandbox.yasandbox.database import mapping

from . import abc

logger = logging.getLogger(__name__)


def list_repr(x):
    # Avoid ugly unicode prefixes in standard list repls, e.g. `[u"foo", u"bar"]`
    return "[{}]".format(", ".join(six.text_type(_) for _ in x))


class UpdateRobotOwners(base.SingletonService):
    """
    Synchronize robot owners from ABC
    """
    tick_interval = 10 * 60

    def tick(self):
        sandbox_users = set(mapping.User.objects.fast_scalar("login"))
        current_owners = {l: rs or [] for l, rs in mapping.RobotOwner.objects.fast_scalar("login", "robots")}

        logger.info("Fetching data from ABC...")

        with context.Timer() as timer:
            robot_owners = abc.ABC().robot_owners()

        logger.info("Done in %.1f sec", timer.secs)

        # Add owners
        new_owners = sandbox_users & (set(robot_owners) - set(current_owners))
        objs = []
        for login in new_owners:
            robots = robot_owners[login]
            objs.append(mapping.RobotOwner(login=login, robots=robots))
            logger.info("User: %s: +%s", login, list_repr(robots))
        if objs:
            mapping.RobotOwner.objects.insert(objs)

        # Delete owners
        non_owners = set(current_owners) - set(robot_owners)
        # Sanity check: do not remove too many robot owners even if ABC mistakenly tells us so.
        if len(non_owners) > len(current_owners) / 2:
            logging.error(
                "ABC states that %d users (out of current %d) do not own robots anymore. "
                "This looks suspicious, ownership will not be removed.",
                len(non_owners), len(current_owners)
            )
        else:
            mapping.RobotOwner.objects.filter(login__in=non_owners).delete()
            for login in non_owners:
                robots = current_owners[login]
                logger.info("User: %s: -%s", login, list_repr(robots))

        # Update owners
        for login, old_robots in six.iteritems(current_owners):
            new_robots = robot_owners.get(login, None)
            if new_robots is None or new_robots == old_robots:
                continue

            mapping.RobotOwner.objects.filter(login=login).update_one(set__robots=new_robots)

            added = sorted(set(new_robots) - set(old_robots))
            deleted = sorted(set(old_robots) - set(new_robots))

            diff = ""
            diff += "+{} ".format(list_repr(added)) if added else ""
            diff += "-{}".format(list_repr(deleted)) if deleted else ""
            logger.info("User: %s: %s", login, diff)
