from __future__ import unicode_literals

import logging
import itertools
import collections

import six
from six.moves import urllib_parse

import concurrent.futures

from sandbox.common import fs
from sandbox.common import rest
from sandbox.common import config
from sandbox.common import patterns
from sandbox.common import itertools as citertools

logger = logging.getLogger(__name__)


class ABC(object):
    """ The controllers reads robots and their owners from ABC """

    URL = "https://abc-back.yandex-team.ru/api/v4"
    RESOURCE_SUPPLIER = 303  # staff
    RESOURCE_TYPE = 53  # robot
    ROLES = [
        1260,  # robots_manager
        1261,  # robots_user
    ]

    @patterns.singleton_property
    def _abc(self):
        oauth_token = fs.read_settings_value_from_file(
            config.Registry().server.auth.oauth.token, ignore_file_existence=True
        )
        return rest.Client(self.URL, auth=oauth_token, logger=logger)

    @staticmethod
    def _read_all(path, params):
        while True:
            data = path.read(**params)
            for item in data.get("results", []):
                yield item
            url = data.get("next")
            if not url:
                break
            up = urllib_parse.urlparse(url)
            params = urllib_parse.parse_qs(up.query)

    def _robots_to_services(self):
        params = {
            "fields": "service.id,resource.name,resource.attributes",
            "state": "granted",
            "supplier": self.RESOURCE_SUPPLIER,
            "type": self.RESOURCE_TYPE,
        }

        robots = collections.defaultdict(list)

        for item in self._read_all(self._abc.resources.consumers["/"], params=params):
            attrs = item["resource"]["attributes"]
            if not any(attr["name"] == "secret_id" for attr in attrs):
                continue
            login = item["resource"]["name"]
            service = item["service"]["id"]
            robots[login].append(service)

        return robots

    def _get_members(self, params):
        return list(self._read_all(self._abc.services.members["/"], params=params))

    def robot_owners(self):
        """
        :return: mapping from Staff logins to the robots they own or can use
        :rtype: Dict[str, List[str]]
        """
        robots_services = self._robots_to_services()
        all_services = itertools.chain.from_iterable(six.itervalues(robots_services))

        params = {"role": self.ROLES, "fields": "person.login,service.id", "service": None}

        owners = collections.defaultdict(set)

        futures = []
        with concurrent.futures.ThreadPoolExecutor(max_workers=16) as pool:
            for group in citertools.grouper(all_services, 30):
                params["service"] = group
                fut = pool.submit(self._get_members, params.copy())
                futures.append(fut)

        for fut in concurrent.futures.as_completed(futures):
            members = fut.result()
            for item in members:
                owners[item["service"]["id"]].add(item["person"]["login"])

        users_robots = collections.defaultdict(set)

        for robot, services in six.iteritems(robots_services):
            for service in services:
                for login in owners[service]:
                    users_robots[login].add(robot)

        return {owner: sorted(robots) for owner, robots in six.iteritems(users_robots)}
