# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import

import logging

from django.conf import settings

from sandbox.step.django_idm_api import validation
from sandbox.step.django_idm_api.compat import get_user_model
from sandbox.step.django_idm_api.exceptions import RoleNotFound, UserNotFound, GroupNotFound, BadRequest

from sandbox.step.statinfra_api.events.models import EventDesc

__all__ = ('BaseHooks', 'AuthHooks', 'RoleNotFound', 'UserNotFound', 'GroupNotFound')


_django_log = logging.getLogger('django')


class RoleStream(object):
    def __init__(self, hooks):
        self.hooks = hooks

    def get_queryset(self):
        return NotImplemented

    def row_as_dict(self, row):
        return NotImplemented


class SuperUsersStream(RoleStream):
    name = 'superusers'
    next_stream = 'memberships'

    def get_queryset(self):
        return get_user_model().objects.filter(is_superuser=True).order_by('pk')

    def values_list(self, queryset):
        return queryset.values_list('pk', 'username')

    def row_as_dict(self, row):
        pk, username = row
        return {
            'login': username,
            'path': '/role/superuser/',
        }


class MembershipStream(RoleStream):
    name = 'memberships'

    def get_queryset(self):
        Membership = get_user_model()._meta.get_field('groups').rel.through
        return Membership.objects.filter(group_id__in=self.hooks.get_group_queryset().values('id')).order_by('pk')

    def values_list(self, queryset):
        return queryset.values_list('pk', 'user__username', 'group__pk')

    def row_as_dict(self, row):
        pk, username, group_pk = row
        return {
            'login': username,
            'path': '/role/group-%d/' % group_pk
        }


class BaseHooks(object):
    _successful_result = {'code': 0}

    def info(self):
        """Срабатывает, когда Управлятор спрашивает систему о том, какие роли она поддерживает."""
        values = self.info_impl()
        return {
            'code': 0,
            'roles': {
                'slug': 'role',
                'name': 'роль',
                'values': {
                    'events': {
                        'name': {'en': 'Events', 'ru': 'События'},
                        'roles': {
                            'slug': 'event_perm',
                            'name': {'en': 'Event permission', 'ru': 'Разрешение на отправку события'},
                            'values': values
                        }
                    },
                    'configs': {
                        'name': {'en': 'Configs', 'ru': 'Конфиги'},
                        'roles': {
                            'slug': 'config_perm',
                            'name': 'Config permission',
                            'values': {}
                        }
                    }
                }
            },
        }

    def add_role(self, login, role, fields):
        """Обрабатывает назначение новой роли пользователю с указанным логином."""
        result = self.add_role_impl(login, role, fields) or {}
        return dict(self._successful_result, **result)

    def add_group_role(self, group, role, fields):
        """Обрабатывает назначение новой роли группе с указанным ID."""
        result = self.add_group_role_impl(group, role, fields) or {}
        return dict(self._successful_result, **result)

    def remove_role(self, login, role, data, is_fired):
        """Вызывается, если у пользователя с указанным логином отзывают роль."""
        try:
            self.remove_role_impl(login, role, data, is_fired)
        except UserNotFound:
            # сотрудник не найден, но мы всё равно возвращаем OK,
            # потому что вдруг это пытаются отозвать доступ кого-то,
            # кого вынесли из админки руками
            pass
        except RoleNotFound:
            # роль неизвестна, но мы всё равно возвращаем OK,
            # потому что вдруг это пытаются отозвать какую-то
            # старую роль
            pass

        return self._successful_result

    def remove_group_role(self, group, role, data, is_deleted):
        """Вызывается, если у группы с указанным ID отзывают роль."""
        try:
            self.remove_group_role_impl(group, role, data, is_deleted)
        except GroupNotFound:
            # группа не найдена, но мы всё равно возвращаем OK,
            # потому что вдруг это пытаются отозвать доступ кого-то,
            # кого вынесли из админки руками
            pass
        except RoleNotFound:
            # роль неизвестна, но мы всё равно возвращаем OK,
            # потому что вдруг это пытаются отозвать какую-то
            # старую роль
            pass
        return self._successful_result

    def get_user_roles(self, login):
        """Вызывается, если Управлятор запрашивает все роли пользователя в системе.
        """
        return dict(
            self._successful_result,
            roles=self.get_user_roles_impl(login),
        )

    def get_all_roles(self):
        """Вызывается для аудита и первичной синхронизации. Deprecated метод.

        Отдает всех пользователей, которые известны в системе, и их роли.
        """
        return {
            'code': 0,
            'users': [
                {
                    'login': login,
                    'roles': roles
                } for login, roles in self.get_all_roles_impl()
            ]
        }

    def get_roles(self, request):
        """Вызывается для аудита и первичной синхронизации. Отдаёт роли пользователей и/или групп постранично"""
        raise NotImplementedError()

    # Абстрактные методы
    def info_impl(self, **kwargs):
        """Возвращает словарь, где ключом является slug роли, а значением
        либо строка с названием роли, либо словарь с расширенным описанием роли.
        """
        raise NotImplementedError()

    def add_role_impl(self, login, role, fields, **kwargs):
        """Выдаёт пользователю с логином login роль role с дополнительными полями fields"""
        raise NotImplementedError()

    def add_group_role_impl(self, group, role, fields, **kwargs):
        """Выдаёт группе с ID group роль role с дополнительными полями fields"""
        raise NotImplementedError()

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        """Отбирает у пользователя login роль role с дополнительными данными data
        и флагом уволенности (is_fired)
            Может бросить UserNotFound и RoleNotFound
        """
        raise NotImplementedError()

    def remove_group_role_impl(self, group, role, data, is_deleted, **kwargs):
        """Отбирает у пользователя login роль role с дополнительными данными data
        и флагом уволенности (is_fired)
            Может бросить UserNotFound и RoleNotFound
        """
        raise NotImplementedError()

    def get_user_roles_impl(self, login, **kwargs):
        """Возвращает список описаний-ролей пользователя с указанным логином.
        """
        raise NotImplementedError()

    def get_all_roles_impl(self, **kwargs):
        """Возвращает список пар (login, список описаний ролей,
        где каждое описание роли это словарь) всех пользователей.
        """
        raise NotImplementedError()


class AuthHooks(BaseHooks):
    """Реализация хуков поверх django.contrib.auth
    """
    def info_impl(self, **kwargs):
        roles = {}
        for ev_desc in EventDesc.objects(list_in_idm=True):
            role = {
                'name': ev_desc.name,
                'fields': [
                    {
                        'slug': field,
                        'name': {
                            'en': 'Parameter %s regex' % field,
                            'ru': 'Регулярное выражение для параметра %s' % field
                        },
                        'type': 'charfield',
                        'required': False
                    } for field in ev_desc.fields
                ]
            }
            roles['%s' % ev_desc.name] = role
        return roles

    def add_role_impl(self, login, role, fields, **kwargs):
        _django_log.info('Add role: %s %s %s', login, role, fields)
        role_type = role['role']

        user, user_created = get_user_model().objects.get_or_create(username=login)

        if role_type == 'events':
            event_name = role['event_perm']
            if event_name in user.event_roles:
                raise BadRequest('User %s already has role %s' % (user.username, role))
            user.is_staff = True
            fields = fields or {}
            user.event_roles[event_name] = fields
            _django_log.info('Now user %s has roles: %s', user.username, user.event_roles)
            user.save()
            return {'data': fields}
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        role_type = role.keys()[0]
        role_value = role[role_type]

        try:
            user = self._get_user(login)
        except UserNotFound:
            raise

        if role_type == 'event_perm':
            event_name = role_value
            if event_name in user.event_roles:
                del user.event_roles[event_name]
                user.save()
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def get_all_roles_impl(self, **kwargs):
        users = (
            get_user_model().objects.
            # filter(Q(groups__id__in=self.get_group_queryset().only('id')) | Q(is_superuser=True)).
            # prefetch_related('groups').
            order_by('username')
            # distinct('username')
        )
        all_roles = []
        for user in users:
            roles = []
            for event_name, event_restrictions in user.event_roles.items():
                roles.append([{'role': 'events', 'event_perm': event_name}, dict(event_restrictions)])

            all_roles.append((user.username, roles))
        return all_roles

    def get_roles(self, request):
        next_url = None
        page_size = getattr(settings, 'ROLES_PAGE_SIZE', 100)
        form = validation.GetRolesForm(request.GET)
        if not form.is_valid():
            form.raise_first()
        type_, since = form.get_clean_data()
        if type_ == 'superusers':
            stream = SuperUsersStream(self)
        else:
            stream = MembershipStream(self)
        qs = stream.get_queryset()
        # maxpk = qs.aggregate(maxpk=Max('pk')).get('maxpk')
        maxpk = qs.order_by('-pk').first()
        qs = stream.values_list(qs)
        if since:
            qs = qs.filter(pk__gt=since)
        qs = qs[:page_size]
        exhausted = maxpk is None
        roles = []
        pk = None

        event_roles_recs = EventDesc.objects(list_in_idm=True).only('name')
        event_roles = ['event-%s' % e.name for e in event_roles_recs]
        roles += event_roles

        for row in qs:
            pk = row[0]
            if pk == maxpk:
                exhausted = True
            roles.append(stream.row_as_dict(row))
        page = {
            'code': 0,
            'roles': roles,
        }
        if exhausted:
            if hasattr(stream, 'next_stream'):
                next_url = '%s?type=%s' % (request.path, stream.next_stream)
        else:
            next_url = '%s?type=%s&since=%d' % (request.path, stream.name, pk)
        if next_url:
            page['next-url'] = next_url
        return page

    # Вспомогательные методы
    def _remove_is_staff_if_needed(self, user):
        if not user.is_superuser and len(user.user_groups) == 0:
            user.is_staff = False

    def _get_user(self, login):
        """Получить пользователя по логину или кинуть эксепшн."""
        try:
            return get_user_model().objects.get(username=login)
        except get_user_model().DoesNotExist:
            raise UserNotFound('User does not exist: %s' % login)

    def _get_group(self, group_id):
        """Получить группу по id или кинуть эксепшн."""
        from sandbox.step.django_mongoengine.mongo_auth.models import UserGroup

        try:
            return self.get_group_queryset().get(pk=int(group_id))
        except UserGroup.DoesNotExist:
            raise RoleNotFound('Group does not exist: %s' % group_id)

    def get_group_queryset(self):
        from sandbox.step.django_mongoengine.mongo_auth.models import UserGroup

        return UserGroup.objects.all()
