# coding: utf-8
from __future__ import unicode_literals, absolute_import

import json

from django import forms
from django.core.validators import EMPTY_VALUES
from sandbox.step.django_idm_api.exceptions import BadRequest


class SelfAwareMixin(object):
    default_error_messages = {
        'required': 'Field "%(name)s" is required.'
    }

    def __init__(self, **kwargs):
        self.name = kwargs.pop('name')
        super(SelfAwareMixin, self).__init__(**kwargs)

    def validate(self, value):
        if value in EMPTY_VALUES and self.required:
            raise forms.ValidationError(self.error_messages['required'] % {'name': self.name})


class JSONField(SelfAwareMixin, forms.Field):
    def clean(self, value):
        if not self.required and value is None:
            value = {}
        else:
            try:
                value = json.loads(value)
            except (ValueError, TypeError):
                raise forms.ValidationError('incorrect json data in field `%(field)s`: %(value)s' % {
                    'field': self.name,
                    'value': value
                })
        return value


class BooleanField(forms.Field):
    def clean(self, value):
        return bool(value and value == '1')


class SelfAwareCharField(SelfAwareMixin, forms.CharField):
    """Поле для более детальных сообщений об ошибках"""


class SelfAwareIntegerField(SelfAwareMixin, forms.IntegerField):
    """Поле для более детальных сообщений об ошибках"""


class InternalForm(forms.Form):
    def get_clean_data(self):
        data = [self.cleaned_data[key] for key in self.base_fields.keys()]
        return data

    def raise_first(self):
        for fieldname in self.base_fields.keys():
            if fieldname in self._errors:
                raise BadRequest(self._errors[fieldname][0])


class AddUserRoleForm(InternalForm):
    login = SelfAwareCharField(name='login', required=True)
    role = JSONField(name='role', required=True)
    fields = JSONField(name='fields', required=False)


class RemoveUserRoleForm(InternalForm):
    login = SelfAwareCharField(name='login', required=True)
    role = JSONField(name='role', required=True)
    data = JSONField(name='data', required=False)
    is_fired = BooleanField()


class AddGroupRoleForm(InternalForm):
    group = SelfAwareIntegerField(name='group', required=True, error_messages={
        'invalid': 'Field "group" must be an integer'
    })
    role = JSONField(name='role', required=True)
    fields = JSONField(name='fields', required=False)


class RemoveGroupRoleForm(InternalForm):
    group = SelfAwareIntegerField(name='group', required=True, error_messages={
        'invalid': 'Field "group" must be an integer'
    })
    role = JSONField(name='role', required=True)
    data = JSONField(name='data', required=False)
    is_deleted = BooleanField()


class GetRolesForm(InternalForm):
    TYPES = ('superusers', 'memberships')

    since = forms.IntegerField(required=False)
    type = forms.ChoiceField(choices=zip(TYPES, TYPES), required=False)

    def get_clean_data(self):
        type_ = self.cleaned_data.get('type') or 'superusers'
        since = self.cleaned_data.get('since')
        return type_, since
