# coding: utf8
"""
Unique identifier for django request.
"""

from __future__ import division, absolute_import, print_function, unicode_literals
import random
import logging


def request_to_id(request, cache='_statbox_django_utils_request_id', **kwa):
    """
    Get a unique identifier for the passed request object.

    Currently, the identifier is semi-random and it is only stable
    because of the caching.
    """

    res = getattr(request, cache, None)
    if res is not None:  # Shortest path: passed req has it.
        return res

    request_passed = request
    request = _get_actual_request(request)

    res = getattr(request, cache, None)
    if res is not None:  # Middle-case: actual request has it.
        setattr(request_passed, cache, res)  # Turn into the shortest path next time.
        return res

    res = request_to_id_base(request, **kwa)
    setattr(request, cache, res)
    if request is not request_passed:
        setattr(request_passed, cache, res)
    return res


def _get_actual_request(request):
    """
    A helper to account for request wrappers such as DRF's.
    """
    try:
        request_actual = getattr(request, '_request')  # e.g. rest_framework
    except Exception:
        request_actual = request

    # An extra check:
    if 'WSGIRequest' in type(request_actual).__name__:
        return request_actual
    else:
        # TODO: catch, check, remove
        logging.getLogger(__name__).log(42, "request._request is not a WSGIRequest")
        return request_actual  # still return it

    return request


def _num_to_string(num):
    """ Something to convert a (large) number to a string, with visual
    considerations. No inter-version stability. """
    return '%x' % (num,)


def request_to_id_base(request):
    # NOTE: another possible way is to hash(hostname + pid + thread_id
    # + id(request), but might not work as well)
    res = '.'.join(str(val) for val in [
        'req',
        _num_to_string(random.getrandbits(4 * 12)),
        _num_to_string(id(request)),
    ])
    return res
