# coding: utf8
"""
Helpers for storing data in a thread-local state (e.g. the request object).


## threadlocals middleware ##

Make the request object everywhere available (e.g. in model instance).

Based on: http://code.djangoproject.com/wiki/CookBookThreadlocalsAndUser
Based on: django-tools

Usage: Add the middleware class
'statbox_django_utils.djangos.thread_local.ThreadLocalMiddleware' to the settings,
obtain the current request object using
`thread_local.get_current_request()` (might return None).
"""

from __future__ import division, absolute_import, print_function, unicode_literals

from threading import local

__all__ = (
    '_thread_locals',
    'get_current_request',
    'get_current_user',
    'ThreadLocalMiddleware',
)


_thread_locals = local()


def get_current_request():
    """ returns the request object for this thead """
    return getattr(_thread_locals, "request", None)


def get_current_user():
    """ returns the current user, if exist, otherwise returns None """
    request = get_current_request()
    if not request:
        return None
    return getattr(request, "user", None)


class ThreadLocalMiddleware(object):
    """ Simple middleware that adds the request object to the thread
    local storage. """

    def process_request(self, request):
        _thread_locals.request = request

    # Make sure the request isn't available after it's done
    def process_response(self, request, response):
        _thread_locals.request = None
        return response

    def process_exception(self, request, exception):
        _thread_locals.request = None
        return
