# coding: utf8

from __future__ import division, absolute_import, print_function, unicode_literals
import logging
from logging import handlers  # noqa
# from .monkeys import logging as logging_monkeys


# Various commonly-useful logging formatters.
common_formatters = {
    'verbose': {
        'format': '[%(asctime)s] %(levelname)s: %(name)s: %(message)s'
    },
    # Extended verbose (timediff, request_id); requires corresponding additions to be used.
    'verbose_ex': {
        'format': (
            '[%(asctime)s](+%(time_diff)5.3fs) uuid=%(django_request_id)s'
            ' %(levelname)s: %(name)s: %(message)s')
    },
    # verbose + thread
    'full': {
        'format': (
            '[%(asctime)s](+%(time_diff)5.3fs)[%(process)d/%(threadName)s]'
            ' uuid=%(django_request_id)s'
            ' %(levelname)s: %(name)s: %(message)s')
    },
    # Non-timestamped messages.
    'simple': {
        'format': '%(name)s: %(levelname)s: %(message)s'
    },
    'console_simple': {
        'format': '[%(asctime)s]: %(message)s'
    },
    # Straightforward.
    # Also used for special-logfiles e.g. with TSKV
    'plain': {
        'format': '%(message)s'
    },
    # record-to-tskv
    'tskv_celery': {
        '()': 'tskv_logging.formatters.TSKVFormatter',
        'tskv_format': 'statinfra-event-processor-celery-log',
        'extra_data': {'codebase': 'statinfra_event_processor'},
    },
    'tskv_django': {
        '()': 'tskv_logging.formatters.TSKVFormatter',
        'tskv_format': 'statinfra-event-processor-django-log',
        'extra_data': {'codebase': 'statinfra_event_processor'},
    },
}


common_filters = {
    'time_diff': {'()': 'tskv_logging.annotators.time_diff_annotator'},
    'django_request_id': {'()': 'sandbox.step.statbox_django_utils.logging.DjangoRequestIdAnnotatingFilter'},
    'hostname': {'()': 'tskv_logging.annotators.full_hostname_annotator'},
    'empty_line': {'()': 'sandbox.step.statbox_django_utils.logging.EmptyLineFilter'}
}

common_annotators = ('time_diff', 'django_request_id', 'hostname', 'empty_line')


class DjangoRequestIdAnnotatingFilter(logging.Filter):
    """
    Annotating Filter to add django_request_id to a logging record (so
    that it can be used in the format string).

    requires 'statbox_django_utils.djangos.thread_local.ThreadLocalMiddleware',
    uses statbox_django_utils.djangos.request_id.
    """

    def filter(self, record):
        request_id = self.get_request_id()
        request_id = request_id or 'unkn'  # placeholder-fallback
        record.django_request_id = request_id
        return True

    def get_request_id(self):
        from .djangos.thread_local import get_current_request
        from .djangos.request_id import request_to_id
        request = get_current_request()
        if not request:
            return
        request_id = request_to_id(request)
        return request_id


class EmptyLineFilter(logging.Filter):
    def filter(self, record):
        return not record.getMessage() == '\n'
