# coding: utf8
"""
Module with all the serializer-building things.
"""

from rest_framework.utils.serializer_helpers import ReturnDict
from rest_framework.serializers import *  # noqa

# TODO: a proper exchaustive __all__ here (because DRF doesn't have it).
# TODO: a ParamListField that does the empty-value check properly.


# https://github.com/sanand0/orderedattrdict/blob/b60d72b8477d9dcadc5552d8a96163230c0d10e7/orderedattrdict/__init__.py#L16
class AttrDictMixin(object):
    '''
    AttrDict extends OrderedDict to provide attribute-style access.
    Items starting with __ or _OrderedDict__ can't be accessed as attributes.
    '''
    __exclude_keys__ = set()

    def __getattr__(self, name):
        'Getting ad.x gets ad["x"]'
        if (name.startswith('__') or name.startswith('_OrderedDict__') or
                name in self.__exclude_keys__):
            return super(AttrDictMixin, self).__getattr__(name)
        else:
            try:
                return self[name]
            except KeyError:
                raise AttributeError(name)

    def __setattr__(self, name, value):
        'Setting ad.x sets ad["x"]'
        if (name.startswith('__') or name.startswith('_OrderedDict__') or
                name in self.__exclude_keys__):
            return super(AttrDictMixin, self).__setattr__(name, value)
        self[name] = value

    def __delattr__(self, name):
        'Deleting ad.x deletes ad["x"]'
        if (name.startswith('__') or name.startswith('_OrderedDict__') or
                name in self.__exclude_keys__):
            return super(AttrDictMixin, self).__delattr__(name)
        del self[name]


class DotReturnDict(AttrDictMixin, ReturnDict):
    __exclude_keys__ = set(['serializer'])


class DotSerializer(Serializer):  # noqa

    @property
    def data(self):
        ret = super(DotSerializer, self).data
        return DotReturnDict(ret, serializer=self)

    @property
    def errors(self):
        ret = super(DotSerializer, self).errors
        return ReturnDict(ret, serializer=self)


class ParamsSerializer(DotSerializer):
    """ A recommended serialiser class with some convenience for
    processing request parameters """
