# coding: utf8

import functools
# from rest_framework.response import Response
# from .serializers import DotReturnDict


__all__ = (
    'ParametrizedViewMixin',
    'browsable_method',
    'parametrized_method',
)


class ParametrizedViewMixin(object):

    def deserialize_params(
            self, serializer_class=None, kwargs=True, cached=False,
            include_body=False, include_files=False, raise_exception=True,
            _data=None, request=None, **ser_kwargs):
        """
        Common function for processing the request's parameters using a serializer.

        :param serializer_class: ..., default: attribute derived from method.
        :param cached: ..., can be set to the cache name to override it.
        """

        if request is None:
            request = self.request

        if serializer_class is None:
            method = request.method.lower()
            serializer_attr_name = 'serializer_{}'.format(method)
            serializer_class = getattr(self, serializer_attr_name)

        if cached:
            cache_target = self
            # NOTE: cache name can be overridden by cached='some_name'.
            cache_name = serializer_class.__name__ if cached is True else cached
            cache_attr_name = '_params_cache__{}'.format(cache_name)
            result = getattr(cache_target, cache_attr_name, None)
            if result is not None:
                return result

        # Gather the whichever things are available

        if kwargs is True:
            try:
                kwargs = self.kwargs
            except Exception:
                kwargs = None

        try:
            get_context = self.get_serializer_context
        except AttributeError:
            context = None
        else:
            context = get_context()

        if _data is None:
            _data = self._get_full_request_data(
                request, kwargs=kwargs,
                include_body=include_body,
                include_files=include_files)

        ser = serializer_class(
            data=_data, context=context, **ser_kwargs)
        ser.is_valid(raise_exception=raise_exception)
        result = ser.data

        if cached:
            setattr(cache_target, cache_attr_name, result)

        return result

    def _get_full_request_data(
            self, request, kwargs=None, include_body=False,
            include_files=False):
        request = self.request if request is None else request

        result = request.query_params.copy()
        if include_body:
            result.update(request.data)  # TODO: avoid adding more values to the list though.
        if include_files:  # XX: possibly obsolete. Needs testing.
            result.update(request.FILES)
        if kwargs:
            result.update(kwargs)

        return result


def _serializer_to_parameters_doc(serializer):
    fields = serializer().get_fields()
    result = []
    for field_name, field in fields.iteritems():
        field_res = dict(
            name=field_name, description=field.help_text,
            required=field.required,
            paramType='query',
            # type='string',
        )
        result.append(field_res)
    return result


def browsable_method(serializer):
    """
    Wrap a method with information about the parameters serializer it uses.

    Currently adds a docstring for swagger.
    """

    def wrap_browsable_method(func):
        wrapped_browsable_method = func

        # # Usefulness of this is unclear.
        # # Unfortunately, DRF doesn't support forms for GET request params.
        # # TODO: check if this would ever be useful for POST/... methods.
        # @functools.wraps(func)
        # def wrapped_browsable_method(*ar, **kwa):
        #     result = func(*ar, **kwa)
        #     if isinstance(result, Response) and isinstance(result.data, dict):
        #         result.data = DotReturnDict(result.data, serializer=serializer)
        #     return result

        import yaml
        params = _serializer_to_parameters_doc(serializer)
        params_str = yaml.safe_dump(dict(parameters=params), default_flow_style=False)
        wrapped_browsable_method.__doc__ = '%s\n---\n%s' % (func.__doc__, params_str)

        return wrapped_browsable_method

    return wrap_browsable_method


def parametrized_method(serializer):
    """
    Wrap a view function to pass deserialized params instead of a
    request to it.

    Additionally applies `browsable_method` wrapper.
    """

    # TODO: use either `decorator`, or `configurable_wrapper` and avoid
    # the extra layer in this.

    def wrap_parametrized_method(func):

        @browsable_method(serializer)
        @functools.wraps(func)
        def wrapped_parametrized_method(self, request, *args, **kwargs):
            params = self.deserialize_params(serializer_class=serializer, request=request)
            kwargs['request'] = request
            kwargs['params'] = params
            return func(*args, **kwargs)

        return wrapped_parametrized_method

    return wrap_parametrized_method
