from __future__ import absolute_import

import logging
import re
import time
from datetime import datetime, timedelta
from multiprocessing.pool import ThreadPool

from mongoengine import Q, NotUniqueError
from pymongo import read_preferences

from .tools import Timer, with_timer, create_task
from .models import Event, ActionConfig, is_free
from .enums import EventState


_log = logging.getLogger('celery.tasks')


@with_timer('Process event {}')
def process_event_wrapper(event_id):
    event = Event.objects.get(id=event_id)
    try:
        event.time_processing_started = datetime.now()
        _log.info('Processing event %s %s % s', event.id, event.name, event.params)
        event_tasks_ids = process_event(event)
    except Exception:
        _log.exception('Failed to process event %s', event_id)
        event.state = EventState.FAILED
        event.save()
    else:
        _log.info('Event %s has been processed', event_id)
        event.state = EventState.PROCESSED
        event.time_processed = datetime.now()
        event.save()
        return event_tasks_ids


def process_event(event):
    action_database = ActionConfig.objects(
        Q(enabled__in=[True, None]) &
        Q(event_deps__name=event.name)
    ).all()

    tasks_ids = []

    for action in action_database:
        asbs_timer = Timer(
            'Check if action should be scheduled. event %s, action %s' % (event.id, action.id)
        ).start()

        response = action_should_be_scheduled(action, event)

        if response['should_be_scheduled']:
            asbs_timer.mark()
            try:
                task = create_task(action, response)
                if task:
                    tasks_ids.append(task.id)
                    _log.info('Task %s for event %s has been created', task.id, event.id)
            except NotUniqueError as ex:
                _log.warning(
                    'Cannot create task for events %r and action %r: %s',
                    [_.id for _ in response['events']], action.id, ex
                )

    return tasks_ids


def action_should_be_scheduled(action, pending_event):
    free_params = action.free_params()
    regex_params = set()
    any_from_list_params = set()

    def _response(should_be_scheduled, params=None, events=None):
        return {
            'should_be_scheduled': should_be_scheduled,
            'params': params,
            'events': events
        }

    def compare(_event_dep, _event):
        if _event_dep['name'] != _event.name:
            return False
        for d_param_name, d_param_value in _event_dep['params'].items():
            if d_param_name not in _event.get_event_params_dict():
                return False
            if is_free(d_param_value):
                continue
            if isinstance(d_param_value, dict) and 'regex' in d_param_value:
                regex = d_param_value['regex']
                if re.match(regex, str(_event.get_param(d_param_name))):
                    regex_params.add(d_param_name)
                    continue
            if isinstance(d_param_value, dict) and 'any' in d_param_value:
                possible_values_list = d_param_value['any']
                if _event.get_param(d_param_name) in possible_values_list:
                    any_from_list_params.add(d_param_name)
                    continue
            if d_param_value != _event.get_param(d_param_name):
                return False
        return True

    event_found = False
    event_index = None
    for i, event_dep in enumerate(action.event_deps):
        if not event_found and compare(event_dep, pending_event):
            event_found = True
            event_index = i
            break

    if not event_found:
        return _response(False)

    action_params = {k: pending_event.get_param(k) for k in free_params}
    action_params.update({k: pending_event.get_param(k) for k in regex_params})
    action_params.update({k: pending_event.get_param(k) for k in any_from_list_params})
    events = [pending_event]

    with Timer(
        'Search of previous suitable events for event %s, num of deps: %s' % (pending_event.id, len(action.event_deps))
    ):
        for i, event_dep in enumerate(action.event_deps):
            if i == event_index:
                continue

            for key in action_params:
                if key not in event_dep['params']:
                    return _response(False)
                if is_free(event_dep['params'][key]):
                    continue
                elif isinstance(event_dep['params'][key], dict):
                    if 'regex' in event_dep['params'][key]:
                        if not re.match(event_dep['params'][key]['regex'], pending_event.get_param(key)):
                            return _response(False)
                    elif 'any' in event_dep['params'][key]:
                        if pending_event.get_param(key) not in event_dep['params'][key]['any']:
                            return _response(False)
                else:
                    if pending_event.get_param(key) != event_dep['params'][key]:
                        return _response(False)

        def _check_event_dep(arg):
            i, event_dep = arg
            if i == event_index:
                return True

            query_dict = {'params__%s' % k: v for k, v in event_dep['params'].items()
                          if k not in free_params}

            query_dict.update({'params__%s' % k: v for k, v in action_params.items()})

            params_cond = []
            for k, v in query_dict.items():
                params_cond.append({'key': k.replace('params__', ''), 'value': v})

            query = dict(
                time_created__lt=pending_event.time_created,
                time_created__gt=pending_event.time_created - timedelta(days=30),
                state__ne=EventState.REJECTED,
                name=event_dep['name'],
                parameters__all=params_cond,
            )
            with Timer('Search dependencies for event {}'.format(pending_event.id)):
                if int(time.time()) % 3:
                    query['read_preference'] = read_preferences.ReadPreference.SECONDARY
                suitable_events = Event.objects(**query).order_by('-time_created').limit(1)

                if not suitable_events:
                    return False

                s_ev = suitable_events.first()
            events.append(s_ev)
            return True

        num_deps = len(action.event_deps)

        if num_deps > 3:
            pool = ThreadPool(num_deps)
            answers = pool.map(_check_event_dep, enumerate(action.event_deps))
        else:
            answers = map(_check_event_dep, enumerate(action.event_deps))

        if not all(answers):
            return _response(False)

    return _response(True, params=action_params, events=events)
