from __future__ import absolute_import

import datetime as dt

import mongoengine as me

from bson.objectid import ObjectId

from ... import settings as conf

from ..enums import ActionTypes


def is_free(value):
    return value is None or value == '~'


class EventDep(me.EmbeddedDocument):
    id = me.ObjectIdField(primary_key=True, default=ObjectId)
    name = me.StringField(required=True)
    params = me.DictField(default=dict)

    def to_mongo(self, use_db_field=True, fields=None):
        return {
            '_id': str(self.id),
            'name': self.name,
            'params': self.params
        }


class ActionConfig(me.Document):
    meta = {'allow_inheritance': True}

    id = me.SequenceField(primary_key=True)
    description = me.StringField(default='')
    action_type = me.StringField()
    event_deps = me.EmbeddedDocumentListField(EventDep)
    semaphores = me.ListField(me.StringField(), default=[])
    custom_params = me.DictField(default=dict)
    enabled = me.BooleanField(default=True)
    author = me.StringField()
    time_created = me.DateTimeField(default=dt.datetime.now)
    time_updated = me.DateTimeField(default=dt.datetime.now)

    def to_dict(self):
        return {
            '_id': self.id,
            'action_type': self.action_type or ActionTypes.SANDBOX_TASK,
            'event_deps': [{'name': ed.name, 'params': ed.params} for ed in self.event_deps],
            'semaphores': self.semaphores,
            'custom_params': self.custom_params,
            'enabled': self.enabled,
            'author': self.author,
            'description': self.description,
        }

    def from_dict(self, data):
        for field in ('description', 'action_type', 'event_deps', 'semaphores', 'custom_params', 'enabled'):
            if field in data:
                setattr(self, field, [EventDep(**_) for _ in data[field]] if field == 'event_deps' else data[field])

    def free_params(self, ev_dep_index=0):
        params = set()
        for k, v in self.event_deps[ev_dep_index].params.items():
            if is_free(v):
                params.add(k)
        return params

    def clean(self):
        free_params = self.free_params()

        if len(self.event_deps) > conf.MAX_NUM_OF_EVENT_DEPS:
            raise me.errors.ValidationError(
                'too many event deps: {num}, max: {max_num}'.format(
                    num=len(self.event_deps),
                    max_num=conf.MAX_NUM_OF_EVENT_DEPS
                )
            )

        for i, ev_dep in enumerate(self.event_deps):
            if len(ev_dep['params']) > conf.MAX_NUM_OF_PARAMS_IN_EVENT:
                raise me.errors.ValidationError(
                    'too many params: {num}, max: {max_num}'.format(
                        num=len(ev_dep['params']),
                        max_num=conf.MAX_NUM_OF_PARAMS_IN_EVENT
                    )
                )
            if self.free_params(i) != free_params:
                raise me.errors.ValidationError('Different sets of free params')


class SandboxActionConfig(ActionConfig):
    task_type = me.StringField(required=True)  # sandbox task type
    task_params = me.DictField(default=dict)  # like owner, priority, etc
    custom_fields = me.DictField(default=dict)  # specific task fields

    meta = {
        'indexes': [
            ('task_type',),
            ('event_deps.name',),
        ]
    }

    def from_dict(self, data):
        for field in ('task_type', 'task_params', 'custom_fields'):
            if field in data:
                setattr(self, field, data[field])
        super(SandboxActionConfig, self).from_dict(data)

    def to_dict(self):
        d = super(SandboxActionConfig, self).to_dict()
        d.update({
            'task_type': self.task_type,
            'task_params': self.task_params,
            'custom_fields': self.custom_fields,
        })
        return d


class SandboxSchedulerConfig(ActionConfig):
    scheduler_id = me.IntField(required=True)

    def from_dict(self, data):
        for field in ('scheduler_id',):
            if field in data:
                setattr(self, field, data[field])
        super(SandboxSchedulerConfig, self).from_dict(data)

    def to_dict(self):
        d = super(SandboxSchedulerConfig, self).to_dict()
        d.update({
            'scheduler_id': self.scheduler_id,
        })
        return d


class WebHookConfig(ActionConfig):
    url = me.StringField(required=True)
    body = me.StringField(default='')
    response_timeout = me.IntField(default=3, max_value=60)
    retry_timeout = me.IntField(default=0, max_value=3600)
    retry_attempts = me.IntField(default=3, max_value=12)

    def from_dict(self, data):
        webhook_data = data.pop('webhook', {})
        for field in ('url', 'body', 'response_timeout', 'retry'):
            if field not in webhook_data:
                continue
            value = webhook_data[field]
            if field == 'retry':
                for subfield in ('timeout', 'attempts'):
                    if subfield not in value:
                        continue
                    setattr(self, '_'.join((field, subfield)), value[subfield])
            else:
                setattr(self, field, value)
        super(WebHookConfig, self).from_dict(data)

    def to_dict(self):
        d = super(WebHookConfig, self).to_dict()
        d.update(
            url=self.url,
            body=self.body,
            response_timeout=self.response_timeout,
            retry=dict(
                timeout=self.retry_timeout,
                attempts=self.retry_attempts
            )
        )
        return d
