from __future__ import absolute_import

import datetime

import mongoengine as me
from bson.objectid import ObjectId

from ..enums import EventState


class EventParam(me.EmbeddedDocument):
    key = me.StringField(required=True)
    value = me.StringField(required=True)


class Event(me.Document):
    # id = me.SequenceField(primary_key=True)
    name = me.StringField(required=True)
    params = me.DictField()
    parameters = me.EmbeddedDocumentListField(EventParam)

    author = me.StringField()

    state = me.StringField(default=EventState.NEW)
    processor_guid = me.UUIDField(binary=False)

    time_processing_started = me.DateTimeField()
    time_created = me.DateTimeField(default=datetime.datetime.now)
    time_processed = me.DateTimeField()

    source_type = me.StringField()  # SANDBOX_TASK, ...
    source = me.DictField(default=dict())

    request_id = me.UUIDField(binary=False)

    meta = {
        'indexes': [
            ('processor_guid',),
            ('state',),
            ('time_created',),
            ('name',),
            ('parameters.key',),
            ('parameters.value',),
            ('parameters.key', 'parameters.value'),
            ('request_id',),
            (
                '-id',
                'state',
                'name',
                'parameters.key',
                'parameters.value',
            ),
        ]
    }

    def to_dict(self):
        return {
            '_id': str(self.id) if isinstance(self.id, ObjectId) else self.id,
            'name': self.name,
            'params': self.params or self.get_event_params_dict(),
            'author': self.author,
            'state': self.state,
            'time_created': str(self.time_created),
            'time_processing_started': str(self.time_processing_started),
            'time_processed': str(self.time_processed),
            'source_type': self.source_type,
            'source': self.source
        }

    def get_param(self, param_key):
        if self.params:
            return self.params[param_key]
        for item in self.parameters:
            if item.key == param_key:
                return item.value

    def get_event_params_list(self):
        return self.parameters or [{'key': k, 'value': str(v)} for k, v in self.params.items()]

    def get_event_params_dict(self):
        return self.params or {item['key']: item['value'] for item in self.parameters}
