from __future__ import absolute_import

import logging
import datetime
import time
import hashlib
import itertools as it

import mongoengine as me

from .action_config import ActionConfig
from .event import Event
from ..enums import TaskState
from .. import http_client

_log = logging.getLogger('celery.tasks')


class Task(me.Document):
    meta = {
        'allow_inheritance': True,
        'indexes': [
            ('state',),
            ('events',),
            ('time_scheduled',),
            ('time_scheduling_started',),
            {
                'fields': ['unique_key'],
                'unique': True,
                'sparse': True,
                'cls': False,
            }
        ]
    }

    event_params = me.DictField()
    semaphores = me.ListField(me.StringField())
    time_created = me.DateTimeField(default=datetime.datetime.now)
    time_scheduling_started = me.DateTimeField()
    time_scheduled = me.DateTimeField()
    state = me.StringField(default=TaskState.NEW)
    custom_params = me.DictField(default=dict)
    events = me.ListField(me.ReferenceField(Event))
    action_config = me.ReferenceField(ActionConfig)
    num_failures = me.IntField(default=0)
    enabled = me.BooleanField(default=True)

    processor_guid = me.UUIDField(binary=False)
    unique_key = me.StringField()

    def to_dict(self):
        def modify(k, v):
            if k == 'id':
                return str(v)
            elif k == 'events':
                return [x.to_dict() for x in v]
            elif k == 'action_config':
                return v.to_dict()
            else:
                return v

        return {k: modify(k, getattr(self, k)) for k in self._fields}

    @property
    def is_important(self):
        return False

    @property
    def last_event(self):
        return max(self.events, key=lambda e: e.id)

    def task_failed_due_to_params(self):
        self.num_failures += 1
        if self.num_failures >= 3 and not self.is_important:  # TODO: to config
            self.action_config.enabled = False
            self.enabled = False
            self.action_config.save()
            _log.warning('Disabling action config %s and task %s', self.action_config.id, self.id)
        self.save()

    def stop_if_running(self):
        raise NotImplementedError()

    def set_state(self, value):
        if self.state == TaskState.REJECTED:
            self.stop_if_running()
        else:
            self.state = value

    def save(self, **kws):
        self.unique_key = hashlib.md5(
            '-'.join(sorted(it.imap(lambda _: str(_.id), it.chain([self.action_config], self.events))))
        ).hexdigest()
        super(Task, self).save(**kws)


class BaseSandboxTask(Task):
    sandbox_task_id = me.IntField()

    def stop_if_running(self):
        _log.info('Stopping task %s', self)
        max_retries = 5
        i = 0
        while self.sandbox_task_id is None and i < max_retries:
            _log.warning('Task %s: waiting for sandbox_task_id', self.id)
            time.sleep(1)
            i += 1
        if self.sandbox_task_id is None:
            _log.warning('Task %s has no sandbox_task_id', self.id)

        sandbox_api = http_client.Sandbox()
        result = sandbox_api.batch.tasks.stop.update([self.sandbox_task_id])
        return result[0]


class SandboxTask(BaseSandboxTask):
    task_type = me.StringField(required=True)
    task_params = me.DictField()
    custom_fields = me.DictField(default=dict)  # specific task fields

    def __repr__(self):
        if self.task_type in ('STATINFRA_TASK', 'STATINFRA_TASK_BETA'):
            return '{id} ({params})'.format(
                id=self.custom_fields.get('action_id'),  # TODO: custom_fields
                params=self.event_params
            )
        return '{task_type} ({params})'.format(
            task_type=self.task_type,
            params=self.event_params
        )

    @property
    def is_important(self):
        return self.task_type in ('STATINFRA_TASK', 'STATINFRA_TASK_BETA')


class SandboxSchedulerTask(BaseSandboxTask):
    scheduler_id = me.IntField(required=True)

    def __repr__(self):
        return 'SCHEDULER %s' % self.scheduler_id


class WebHookTask(Task):
    url = me.StringField(required=True)
    body = me.StringField()
    response_timeout = me.IntField()
    first_attempt = me.DateTimeField()
    attempts = me.IntField(default=0)
    retry_timeout = me.IntField()
    retry_attempts = me.IntField()

    def __repr__(self):
        return 'WEB_HOOK {} ({})'.format(self.url, str(self.pk))

    def stop_if_running(self):
        pass
