from __future__ import absolute_import

import re
import json
import uuid
import logging
import datetime as dt

import jinja2
import requests
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter

from mongoengine import QuerySet, DoesNotExist
from bson.objectid import ObjectId

from .enums import ActionTypes, TaskState
from .models import SandboxTask, SandboxSchedulerTask, Event, Task, ActionConfig, EventDesc, WebHookTask


_log = logging.getLogger('celery.tasks')
_django_log = logging.getLogger('django')


def no_semaphore_semaphore(s):
    if s.startswith('external:'):
        return ['no-semaphore/' + (s[:s.find('/', s.find('/') + 1)] if s.count('/') > 1 else s)]
    else:
        return ['no-semaphore/%s' % s]


def get_seconds_from_scale(scale):
    m = re.match('(\d+)(\w+)', scale)
    if m is None:
        raise Exception('Wrong scale: %s' % scale)
    num, unit = m.groups()
    mult_coeff = {
        'm': 60,
        'h': 60 * 60,
        'd': 60 * 60 * 24
    }[unit]

    return int(num) * mult_coeff


def round_time(time_obj, scale):
    round_to = get_seconds_from_scale(scale)
    midnight = time_obj.replace(hour=0, minute=0, second=0, microsecond=0)
    seconds = (time_obj - midnight).seconds
    rounded = midnight + dt.timedelta(seconds=(seconds // round_to) * round_to)
    return rounded


def is_for_scale(time_obj, scale):
    round_to = get_seconds_from_scale(scale)
    midnight = time_obj.replace(hour=0, minute=0, second=0, microsecond=0)
    seconds = (time_obj - midnight).seconds
    return seconds % round_to == 0


def get_requests_session_with_retries(host, retries_num=10, backoff_factor=2):
    session = requests.Session()
    retries = Retry(total=3, backoff_factor=0.5)
    session.mount(host, HTTPAdapter(max_retries=retries))
    return session


class Timer(object):
    def __init__(self, global_message):
        self.global_message = global_message
        self.start_time = None
        self.uuid = uuid.uuid4()

    def start(self):
        self.start_time = dt.datetime.now()
        return self

    def mark(self, message='mark'):
        delta = dt.datetime.now() - self.start_time

        _log.info(
            'Timer {uuid}: {glob_msg} - {msg} - {tdelta} sec'.format(
                uuid=self.uuid,
                glob_msg=self.global_message,
                msg=message,
                tdelta=delta.total_seconds()
            )
        )

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, *args):
        self.mark()


def with_timer(global_message):  # TODO: check with scheduled tasks
    def deco(f):
        def wrapper(*args, **kwargs):
            with Timer(global_message.format(*args, **kwargs)):
                return f(*args, **kwargs)
        return wrapper
    return deco


def split_to_chunks(items, chunk_size):
    return [items[i:i + chunk_size]
            for i in xrange(0, len(items), chunk_size)]


def if_true(cond):
    def deco(f):
        def wrapper(*args, **kwargs):
            if cond is True:
                return f(*args, **kwargs)
        return wrapper
    return deco


def create_task(action, asbs_response, do_not_save=False, raise_errors=False):
    task = None
    action_type = action.action_type or ActionTypes.SANDBOX_TASK
    if action_type == ActionTypes.SANDBOX_TASK:
        task = SandboxTask(
            task_type=action.task_type,
            task_params=action.task_params,
            event_params=asbs_response['params'],
            custom_fields=action.custom_fields,
            semaphores=action.semaphores,
            custom_params=action.custom_params,
            events=asbs_response['events'],
            action_config=action,
        )
        description = task.task_params.get('description')
        if description is not None:
            task.task_params['description'] = description.format(**asbs_response['params'])
    elif action_type == ActionTypes.SANDBOX_SCHEDULER:
        task = SandboxSchedulerTask(
            scheduler_id=action.scheduler_id,
            event_params=asbs_response['params'],
            semaphores=action.semaphores,
            custom_params=action.custom_params,
            events=asbs_response['events'],
            action_config=action,
        )
    elif action_type == ActionTypes.WEB_HOOK:
        events = asbs_response['events']
        event_params = asbs_response['params']
        body = action.body
        try:
            if body:
                template = jinja2.Template(body)
                body = template.render(events=[e.to_dict() for e in events], params=event_params)
                json.loads(body)
                if isinstance(body, dict):
                    raise ValueError('Template result must be a dict')
        except (jinja2.TemplateError, TypeError, ValueError) as ex:
            msg = 'Error while rendering or parsing template: {}\n\nbody: {!r}\n\nparams: {!r})'.format(
                ex, body, event_params
            )
            _log.error(msg)
            if raise_errors:
                raise ValueError(msg)
        else:
            task = WebHookTask(
                event_params=event_params,
                events=events,
                action_config=action,
                url=action.url,
                body=body,
                response_timeout=action.response_timeout,
                retry_timeout=action.retry_timeout,
                retry_attempts=action.retry_attempts
            )
    if task is not None and not do_not_save:
        task.save()
    return task


def query_to_py(q):

    def _elem_to_py(el):
        if hasattr(el, 'to_dict'):
            d = el.to_dict()
        elif hasattr(el, 'to_mongo'):
            d = el.to_mongo()
        else:
            d = el
        if '_id' in d and isinstance(d['_id'], ObjectId):
            d['_id'] = str(d['_id'])
        elif 'id' in d and isinstance(d['id'], ObjectId):
            d['id'] = str(d['id'])
        return d

    if isinstance(q, (QuerySet, list, tuple)):
        return [_elem_to_py(x) for x in q]
    else:
        return _elem_to_py(q)


def generate_stats(dtime_min, dtime_max):
    sensors = dict()

    sensors['total_configs_count'] = ActionConfig.objects.count()

    sensors['events_count'] = Event.objects(
        time_created__gt=dtime_min,
        time_created__lt=dtime_max
    ).count()

    scheduled_tasks = Task.objects(
        state=TaskState.SCHEDULED,
        time_scheduled__gt=dtime_min,
        time_scheduled__lt=dtime_max
    )

    scheduled_tasks_count = scheduled_tasks.count()
    sensors['scheduled_tasks_count'] = scheduled_tasks_count
    reaction_time_list = [
        (t.time_scheduled - t.last_event.time_created).total_seconds()
        for t in scheduled_tasks
    ]

    if scheduled_tasks_count != 0:
        sensors['avg_reaction_time'] = float(sum(reaction_time_list)) / scheduled_tasks_count
        sensors['min_reaction_time'] = min(reaction_time_list)
        sensors['max_reaction_time'] = max(reaction_time_list)

    return sensors


def _check_event_params(event, event_role):
    event_params_dict = event['params']
    for param_name, value_re in event_role.items():
        if param_name in event_params_dict \
                and event_params_dict[param_name] \
                and not re.match(value_re, event_params_dict[param_name]):
            return False

    return True


def user_can_manage_event(user, event):
    event_desc = None
    try:
        event_desc = EventDesc.objects.get(name=event['name'])
    except DoesNotExist:
        return False, 'no event description'

    if event_desc.public:
        if event_desc.list_in_idm:
            if user and hasattr(user, 'event_roles'):
                if event['name'] not in user.event_roles \
                        or not _check_event_params(event, user.event_roles.get(event['name'], {})):
                    _django_log.warning(
                        'User %s is creating public event %s shown in idm without granted role',
                        user.username,
                        event['name']
                    )
            else:
                _django_log.warning(
                    'Anonymous user is creating public event %s shown in idm',
                    event['name']
                )

        return True, 'ok'

    else:
        if not user:
            return False, 'non-public event and anonymous user'

        if event_desc.author == user.username:
            return True, 'ok'

        if hasattr(user, 'user_events') and event['name'] in user.event_roles \
                and _check_event_params(event, user.event_roles.get(event['name'], {})):
            return True, 'ok'

    return False, 'not authorized'


def create_event_params_list(params):
    return [{'key': k, 'value': str(v)} for k, v in params.items()]
