from __future__ import absolute_import

import logging
import datetime as dt

from rest_framework.response import Response
from rest_framework.decorators import api_view
from rest_framework import status

from mongoengine.errors import ValidationError, DoesNotExist, FieldDoesNotExist, OperationError

from sandbox.step.statinfra_api.common import CommonViewMixin

from .models.event import Event
from .models.action_config import ActionConfig, SandboxActionConfig, SandboxSchedulerConfig, WebHookConfig
from .models.task import Task
from .models.event_registry import EventDesc
from .models.event_story import EventStory

from .enums import TaskState, ActionTypes
from .tools import query_to_py, user_can_manage_event, create_event_params_list, create_task
from .tasks import travel_through_the_event

from .. import settings as conf


_django_log = logging.getLogger('django')


class EventsCollection(CommonViewMixin):

    def post(self, request):
        data = request.data
        username = '[unauthenticated]'
        try:
            user = getattr(request, 'yauser')
            if user.uid:
                username = user.login
        except Exception:
            _django_log.exception('Failed to detect user')

        source_type = data.get('source_type')
        source = data.get('source', {})

        user = getattr(request, 'user')

        request_id = data.get('request_id')
        if request_id:
            existing_events = Event.objects(request_id=request_id)
            if existing_events:
                _django_log.warning('Events with request_id %s already exist', request_id)
                return Response({'ids': [str(x.id) for x in existing_events]})

        forbidden_events = set()
        for ev in data['events']:
            ev_name = ev['name']
            has_permissions, reason = user_can_manage_event(user, ev)
            if not has_permissions:
                _django_log.warning(
                    'User %s cannot create event %s; reason: %s',
                    username, ev_name, reason
                )
                forbidden_events.add((ev_name, reason))

        if conf.USE_ACL:
            if forbidden_events:
                return Response(
                    {'message': 'User {} cannot create events {}'.format(
                        username,
                        ', '.join(['%s (%s)' % (m[0], m[1]) for m in forbidden_events])
                    )},
                    status=403
                )

        evs = []
        for ev in data['events']:
            evs.append(
                Event(
                    name=ev['name'],
                    parameters=create_event_params_list(ev.get('params', {})),
                    author=username,
                    source_type=source_type,
                    source=source,
                    request_id=request_id,
                )
            )
        if evs:
            objects = Event.objects.insert(evs)
            return Response({'ids': [str(x.id) for x in objects]})
        else:
            return Response({'ids': []})

    def get(self, request):
        skip = int(request.GET.get('skip', 0))
        limit = int(request.GET.get('limit', 30))
        sort = request.GET.get('sort')
        if sort:
            sort = sort.strip()
            if sort == '_id':
                sort = 'time_created'
            elif sort == '-_id':
                sort = '-time_created'

        event_name = request.GET.get('name')
        event_author = request.GET.get('author')
        event_age = int(request.GET.get('age', 10))
        event_params = {}
        for k, v in request.GET.items():
            if k.startswith('params__'):
                event_params[k] = v
        state = request.GET.get('state')

        cond = {'time_created__gt': dt.datetime.now() - dt.timedelta(days=event_age)}
        if event_name is not None:
            cond['name'] = event_name
        if event_author is not None:
            cond['author'] = event_author
        if event_params:
            params_cond = []
            for k, v in event_params.items():
                params_cond.append({'key': k.replace('params__', ''), 'value': v})
            cond['parameters__all'] = params_cond

        if state:
            cond["state"] = state

        q = Event.objects(**cond).skip(skip)

        if sort:
            if 'params__' in sort:
                field_name = sort.split('__')[-1]
                desc = sort.startswith('-')
                sort_order = -1 if desc else 1
                pipeline = [
                    {'$match': {'parameters.key': field_name}},
                    {'$unwind': '$parameters'},
                    {'$match': {'parameters.key': field_name}},
                    {'$sort': {'parameters.value': sort_order}}
                ]
                if limit != -1:
                    pipeline.append({'$limit': limit})
                if skip:
                    pipeline.append({'$skip': skip})
                q = list(q.aggregate(*pipeline))
                ids = [x['_id'] for x in q]

                res = Event.objects(id__in=ids).all()
                res = sorted(res, key=lambda item: ids.index(item.id))

                res = query_to_py(res)
                return Response({'result': res})

            else:
                q = q.order_by(sort)

        if limit != -1:
            q = q.limit(limit)
        res = query_to_py(q)
        return Response({'result': res})


class EventsItem(CommonViewMixin):

    def get(self, request, event_id):
        res = query_to_py(Event.objects.get(id=event_id))
        return Response({'result': res})


class ActionConfigCollection(CommonViewMixin):

    def post(self, request):
        username = ''
        try:
            user = getattr(request, 'yauser')
            if user.uid:
                username = user.login
        except Exception:
            _django_log.exception('Failed to detect user')

        data = request.data
        ac = None
        action_type = data.get('action_type', ActionTypes.SANDBOX_TASK)
        try:
            if action_type == ActionTypes.SANDBOX_TASK:
                ac = SandboxActionConfig(**data)
            elif action_type == ActionTypes.SANDBOX_SCHEDULER:
                ac = SandboxSchedulerConfig(**data)
            elif action_type == ActionTypes.WEB_HOOK:
                ac = WebHookConfig()
                ac.from_dict(data)
            ac.author = username
            ac.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        return Response({'id': ac.id})

    def get(self, request):
        skip = int(request.GET.get('skip', 0))
        limit = int(request.GET.get('limit', 30))
        sort = request.GET.get('sort')

        filter_params = {
            k: request.GET.get(k) for k in request.GET
            if k not in ('skip', 'limit', 'sort', 'description')
        }
        for item in ('enabled',):
            if item in filter_params:
                filter_params[item] = bool(int(filter_params[item]))
        acs = ActionConfig.objects(**filter_params)
        if 'description' in request.GET:
            description = request.GET.get('description')
            acs = acs.filter(description__icontains=description)

        acs = acs.order_by(sort).skip(skip)
        if limit != -1:
            acs = acs.limit(limit)
        res = query_to_py(acs)
        return Response({'result': res})


class ActionConfigTryWebhook(CommonViewMixin):

    def post(self, request):
        data = request.data
        action_type = data.get('action_type', ActionTypes.SANDBOX_TASK)
        try:
            if action_type != ActionTypes.WEB_HOOK:
                raise TypeError('Accepted action type {} only'.format(ActionTypes.WEB_HOOK))
            action = WebHookConfig()
            action.from_dict(data)
            events = [
                Event(
                    name=dep.name,
                    parameters=create_event_params_list({k: request.GET.get(k, v) for k, v in dep.params.iteritems()}),
                )
                for dep in action.event_deps
            ]
            params = {_: request.GET.get(_, "~") for _ in action.free_params()}
            task = create_task(action, {'events': events, 'params': params}, do_not_save=True, raise_errors=True)
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except Exception as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        return Response({'body': task.body})


class ActionConfigItem(CommonViewMixin):

    ALLOWED_PARAMS = [
        'action_type', 'event_deps', 'semaphores', 'custom_params', 'enabled',
        'task_params', 'custom_fields', 'description', 'webhook'
    ]

    def get(self, request, ac_id):
        res = query_to_py(ActionConfig.objects.get(id=ac_id))
        return Response({'result': res})

    def put(self, request, ac_id):
        data = request.data
        data = {k: v for k, v in data.items() if k in ActionConfigItem.ALLOWED_PARAMS}
        try:
            doc = ActionConfig.objects.with_id(ac_id)
            if doc:
                doc.from_dict(data)
                doc.time_updated = dt.datetime.now()
                doc.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        return Response()

    def delete(self, request, ac_id):
        ActionConfig.objects.get(id=ac_id).delete()
        return Response()


@api_view(['GET'])
def last_tasks_info(request):
    num = int(request.GET.get('num', 30))
    action_id = request.GET.get('action_id')
    if action_id is not None:
        objects = Task.objects.filter(task_params__action_id__contains=action_id)
    else:
        objects = Task.objects
    last_processed_tasks = objects.filter(state=TaskState.SCHEDULED).order_by('-id')[:num]
    return Response([
        {
            'task': repr(t),
            'time_taken': t.time_scheduled - max(t.events, key=lambda o: o.id).time_created,
            'time_scheduled': t.time_scheduled
        } for t in last_processed_tasks
    ])


@api_view(['GET'])
def slow_tasks_top(request):
    num = int(request.GET.get('num', 300))
    top_size = int(request.GET.get('top_size', 5))
    last_processed_tasks = Task.objects.filter(state=TaskState.SCHEDULED).order_by('-time_created')[:num]
    return Response(sorted([
        {
            'task': repr(t),
            'time_taken': t.time_scheduled - max(t.events, key=lambda o: o.id).time_created,
            'time_scheduled': t.time_scheduled
        } for t in last_processed_tasks
    ], key=lambda x: -x['time_taken'])[:top_size])


@api_view(['GET', 'POST'])
def protected_view(request):
    return Response('Deprecated', status=400)


class EventDescriptionCollection(CommonViewMixin):
    def post(self, request):
        data = request.data
        try:
            ev_desc = EventDesc(**data)
            ev_desc.author = request.yauser.login
            ev_desc.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        return Response()

    def get(self, request):
        filter_params = {k: request.GET.get(k) for k in request.GET}
        for item in ('public', 'list_in_idm'):
            if item in filter_params:
                filter_params[item] = bool(int(filter_params[item]))

        ev_descs = EventDesc.objects(**filter_params)
        res = query_to_py(ev_descs)
        return Response({'result': res})


class EventDescriptionItem(CommonViewMixin):
    def put(self, request, name):
        ev_desc = EventDesc.objects.get(name=name)
        if request.user.username != ev_desc.author:
            return Response(status=403)
        data = request.data
        try:
            ev_desc.update(**data)
            ev_desc.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        return Response()

    def get(self, request, name):
        try:
            ev_desc = EventDesc.objects.get(name=name)
        except DoesNotExist:
            return Response(status=404)
        res = query_to_py(ev_desc)
        return Response({'result': res})

    def delete(self, request, name):
        ev_desc = EventDesc.objects.get(name=name)
        if request.user.username != ev_desc.author:
            return Response(status=403)
        ev_desc.delete()
        return Response()


class RejectEvent(CommonViewMixin):
    def post(self, request):
        data = request.data
        event_id = data['id']
        try:
            event_story = EventStory()
            event_story.initial_event = Event.objects.get(id=event_id)
            event_story.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        travel_through_the_event.delay(
            event_id,
            event_story=event_story,
            reject=True
        )
        return Response({'id': str(event_story.id)})


class EventStoryItem(CommonViewMixin):

    # get_event_story
    def get(self, request, story_id):
        res = EventStory.objects.get(id=story_id).to_dict()
        return Response({'result': res})


class EventStoryCollection(CommonViewMixin):

    # request_event_story
    def post(self, request):
        data = request.data
        event_id = data['id']
        try:
            event_story = EventStory()
            event_story.initial_event = Event.objects.get(id=event_id)
            event_story.save()
        except ValidationError as e:
            return Response({'message': e.to_dict()}, status=status.HTTP_400_BAD_REQUEST)
        except (TypeError, FieldDoesNotExist, OperationError) as e:
            return Response({'message': str(e)}, status=status.HTTP_400_BAD_REQUEST)
        travel_through_the_event.delay(
            event_id,
            event_story=event_story,
            reject=False
        )
        return Response({'id': str(event_story.id)})
