#!/usr/bin/env python
# coding: utf8
"""
Yet another version of timetail.

 * Does not require last line to be non-junk
 * Reads the file in reverse chunks
   * WARN: Reads stdin into memory.
"""
from __future__ import absolute_import


import os
import sys
import argparse
import datetime
import re
import itertools
import logging

from pyaux import reversed_lines


_log = logging.getLogger(__name__)


def timetail_g(line_iter, params, now=None):
    now = now or datetime.datetime.now()

    if params.dtfmt == 'auto':
        import dateutil.parser
        dtparse = dateutil.parser.parse
    else:
        def dtparse_format(st):
            return datetime.datetime.strptime(st, params.dtfmt)

        dtparse = dtparse_format

    dt_re = re.compile(params.regex)

    unmatches = []
    found_time = False

    # Allow for empty file:
    if params.allow_empty:
        try:
            first_line = next(line_iter)
        except StopIteration:
            return
        # Put it back
        line_iter = itertools.chain([first_line], line_iter)

    for line in line_iter:
        match = dt_re.search(line)
        if not match:
            _log.debug("Ignoring unmatched line: %r", line)
            unmatches.append(line)
            continue
        grp = match.groups()
        if not grp:
            raise Exception("Supplied regex contains no groups")
        dt_str = grp[0]
        try:
            dt = dtparse(dt_str)
        except Exception as exc:
            # raise
            _log.warning("Matched but unparsed datetime: %r  (%r)", dt_str, exc)
            continue

        found_time = True  # Not a total failure

        time_diff = (now - dt).total_seconds()
        if time_diff > params.backtime:
            _log.debug("Found a line %rs back (backtime %rs), exiting", time_diff, params.backtime)
            return

        _log.info("Found a datetime within bounds: %r / %r < %r", dt, time_diff, params.backtime)

        # Otherwise we're still within the requested time boundaries
        if params.include_unmatched:
            for unmatch in unmatches:
                yield unmatch
            unmatches = []
        yield line  # ... the main line

    if not found_time:  # ... total failure.
        raise Exception("No matching lines found at all")


def timetail(*ar, **kwa):
    res = timetail_g(*ar, **kwa)
    res = list(res)
    res = reversed(res)
    res = ('%s\n' % (l,) for l in res)
    return ''.join(res)


def run_on_fo(fo_or_filename, params, func=None):
    if func is None:
        def func_default(li):
            return timetail(li, params)

        func = func_default

    if isinstance(fo_or_filename, (bytes, unicode)):
        close_fo = True
        try:
            fo = open(fo_or_filename, 'r')
        except IOError as e:
            if (e.errno == os.errno.ENOENT) and params.allow_empty:
                # Allow silently skipping non-existing files if asked to.
                # return ''  # NOTE: function-dependent
                # Non-function-dependent:
                close_fo = False
                line_iter = iter([])  # Handle as empty
            else:
                raise
        else:
            line_iter = reversed_lines(fo)
    else:
        close_fo = False
        fo = fo_or_filename
        # We are assuming that we can't seek an arbitrary file object
        data = fo.read()
        line_iter = reversed(data.splitlines())
    try:
        return func(line_iter)
    finally:
        if close_fo:
            fo.close()


def _parse_args(args=None):
    parser = argparse.ArgumentParser(description='`tail` that works on seconds instead of lines.')
    parser.add_argument('filename', help='log file to run on; `-` for stdin')
    parser.add_argument(
        '-n', dest='backtime', type=int, default=300,
        help='Seconds count from now() to look at from the end of the <log file>')
    parser.add_argument(
        '-d', dest='debug', action='store_true', default=False,
        help='Print the debugging output on stderr')
    parser.add_argument(
        '-z', dest='nonzero', action='store_true', default=False,
        help='Expect zero output i.e. exit with non-zero status on non-empty output')
    parser.add_argument(
        '-s', dest='allow_empty', action='store_true', default=False,
        help='Allow the specified file to not exist or to be empty (`test -s ...`)')
    parser.add_argument(
        '-r', dest='regex',
        default=r'(?:^|[^\d])(\d{4}.?\d{2}.?\d{2}..?\d{2}.?\d{2}.?\d{2})(?:[^\d]|$)',
        help='Regexp to pick timestamp from string ($1 must select timestamp)',
    )
    parser.add_argument(
        '--dtfmt', dest='dtfmt', default='auto',
        help='Datetime format. Default: "auto" (use dateutil)')
    parser.add_argument(
        '--include-unmatched', dest='include_unmatched',
        action='store_true', default=False,
        help='Include lines without timestamp after a line with timestamp.')
    args = parser.parse_args(args=args)
    return args


def main(args=None, argv=None):
    if args is None:
        args = _parse_args(argv or sys.argv[1:])
    if args.filename == '-':
        fo = sys.stdin
    else:
        fo = args.filename

    if args.debug:
        logging.basicConfig(level=10, stream=sys.stderr)

    try:
        res = run_on_fo(fo, args)
    except Exception as e:
        print e
        return 1
    else:
        sys.stdout.write(res)
        if args.nonzero:
            if res:
                return 1


if __name__ == '__main__':
    sys.exit(main())
