import abc
import json

import six

from sandbox.common import patterns as common_patterns

from sandbox.yasandbox.database import mapping


class ParametersUpdateResult(common_patterns.Abstract):
    __slots__ = ("errors", "required", "input_updated", "output_updated")
    __defs__ = (None,) * 5


@six.add_metaclass(abc.ABCMeta)
class Serializer(object):
    def __call__(self, *args, **kwargs):
        return self

    @abc.abstractmethod
    def encode(self, value):
        pass

    @abc.abstractmethod
    def decode(self, value):
        pass


class IdentitySerializer(Serializer):

    @classmethod
    def encode(cls, value):
        return value

    @classmethod
    def decode(cls, value):
        return value


class ModelSerializer(Serializer):
    MODEL_CLS = None

    def __init__(self, model_cls=None):
        self.model_cls = model_cls or self.MODEL_CLS

    def encode(self, value):
        return value.to_json()

    def decode(self, value):
        return self.model_cls.from_json(value)


class ListModelSerializer(ModelSerializer):
    def encode(self, value):
        assert isinstance(value, list)
        return list(map(lambda _: _.to_json(), value))

    def decode(self, value):
        assert isinstance(value, list)
        return list(map(self.model_cls.from_json, value))


class ParametersMetaSerializer(ModelSerializer):
    MODEL_CLS = mapping.ParametersMeta


class TemplateParametersMeta(ListModelSerializer):
    MODEL_CLS = mapping.TaskTemplate.Task.ParameterMeta


class ClientListSerializer(ListModelSerializer):
    MODEL_CLS = mapping.Client

    def decode(self, value):
        assert isinstance(value, list)
        return list(map(mapping.base.LiteDocument.generate_field_lite_class(self.model_cls).from_json, value))


class ReportsInfoSerializer(ListModelSerializer):
    MODEL_CLS = mapping.Template.ReportInfo


class ParametersUpdateSerializer(Serializer):

    def encode(self, value):
        return dict(value)

    def decode(self, value):
        assert isinstance(value, dict)
        return ParametersUpdateResult(**value)


class TaskboxRequest(object):
    """ Request that is sent from server to worker to perform bridge method. """

    def __init__(
        self, method_name, model=None, task_type=None, read_only=False,
        arg=None, arg_serializer=None, result_serializer=None, request_id=None, age=None,
    ):
        self.method_name = method_name
        self.model = model
        self.task_type = task_type
        self.read_only = read_only
        self.arg = arg
        self.arg_serializer = arg_serializer or IdentitySerializer()
        self.result_serializer = result_serializer or IdentitySerializer()
        self.request_id = request_id
        self.age = age

    def encode(self):
        if bool(self.model) == bool(self.task_type):
            raise ValueError("Exactly one parameter (model xor task_type) must be defined")
        if self.model and not isinstance(self.model, (mapping.Template, mapping.base.LiteDocument)):
            raise ValueError("Model {!r} bust be an instance of `mapping.Template`".format(self.model))

        model_cls_name = None
        if self.model is not None:
            model_cls_name = (
                self.model.__class__.__name__
                if isinstance(self.model, mapping.Template) else
                self.model.__document_type__.__name__
            )

        model_json = None
        if self.model:
            model_json = self.model.to_json()
            # TODO: remove after tasks binaries with age < 9 will be disabled
            if self.age and self.age < 9:
                model_obj = json.loads(model_json)
                sem_acquires = model_obj.get("req", {}).get("sem", {}).get("acquires")
                if sem_acquires:
                    for item in sem_acquires:
                        item.pop("public", None)
                    model_json = json.dumps(model_obj)
        return json.dumps({
            "method": self.method_name,
            "model": model_json,
            "cls": model_cls_name,
            "task_type": self.task_type,
            "read_only": bool(self.read_only or self.task_type),
            "arg": self.arg_serializer.encode(self.arg),
            "arg_serializer": type(self.arg_serializer).__name__,
            "result_serializer": type(self.result_serializer).__name__,
            "rid": self.request_id,
        })

    @staticmethod
    def __find_serializer(name):
        serializer = globals()[name]
        if not issubclass(serializer, Serializer):
            raise ValueError("Serializer class must be subclass of `protocol.Serializer`")
        return serializer()

    @classmethod
    def decode(cls, s):
        """
        :type s: str
        :rtype: TaskboxRequest
        """
        obj = json.loads(s)
        model_cls_name = obj["cls"]
        model_cls = getattr(mapping, model_cls_name) if model_cls_name else None
        arg_serializer = cls.__find_serializer(obj["arg_serializer"])
        result_serializer = cls.__find_serializer(obj["result_serializer"])
        return cls(
            obj["method"],
            model=(
                mapping.base.LiteDocument.generate_field_lite_class(model_cls).from_json(obj["model"])
                if model_cls else
                None
            ),
            task_type=obj["task_type"],
            read_only=obj["read_only"],
            arg=arg_serializer.decode(obj["arg"]),
            arg_serializer=arg_serializer,
            result_serializer=result_serializer,
            request_id=obj.get("rid"),
        )

    @property
    def model_serializer(self):
        return ModelSerializer(self.model.__class__) if self.model else IdentitySerializer()


class TaskboxResponse(object):
    """ Response from Taskbox that is created in worker based on tasks binary. """

    def __init__(self, model, changed_fields, result, exc_info):
        self.model = model
        self.changed_fields = changed_fields
        self.result = result
        self.exc_info = exc_info

    def encode(self):
        return json.dumps({
            "model": self.model.to_json() if self.model else None,
            "changed_fields": self.changed_fields,
            "result": self.result,
            "exc_info": self.exc_info,
        })

    @classmethod
    def decode(cls, s, request):
        """
        :type s: str
        :type request: TaskboxRequest
        :rtype: TaskboxResponse
        """
        obj = json.loads(s)
        model = obj["model"]
        return cls(
            request.model_serializer.decode(model) if model else None,
            obj["changed_fields"],
            request.result_serializer.decode(obj["result"]),
            obj["exc_info"],
        )
