import abc
import json
import logging

import six

from sandbox.common import itertools as common_itertools
import sandbox.common.types.misc as ctm
import sandbox.common.joint.client as jclient

from sandbox.web import response
from sandbox.yasandbox.database import mapping

from sandbox.taskbox import config
from sandbox.taskbox import errors

from . import protocol


@six.add_metaclass(abc.ABCMeta)
class ServiceClient(jclient.BaseServiceClient):
    """ Wrapper for the Taskbox service's RPC client. It retries and restores the connection implicitly. """

    def __init__(self, rpc_client_args, logger=None):
        super(ServiceClient, self).__init__(logger or logging.getLogger(__name__))
        self._config = getattr(config.Registry().taskbox, self.service_name)
        self._srv = jclient.RPCClient(self._config.rpc, *rpc_client_args)

    @abc.abstractproperty
    def service_name(self):
        return ""


class BackQueryHandler(object):
    def __init__(self, task=None):
        self.task = task

    def handle_request(self, rtype, data):
        handler = {
            ctm.TaskboxBackQuery.RESOURCES: self.resources,
            ctm.TaskboxBackQuery.CREATE_RESOURCES: self.create_resources,
            ctm.TaskboxBackQuery.TASKS: self.tasks,
            ctm.TaskboxBackQuery.API_REQUEST: self.api_request,
        }.get(rtype)
        if not handler:
            raise errors.UnknownBackQueryType(rtype)
        return handler(data)

    def create_resources(self, data):
        # Use only to create resource in on_enqueue
        from sandbox.yasandbox import controller
        import yasandbox.api.json.resource as api

        from sandbox import sdk2

        data.setdefault("attrs", {})
        if data.get("for_parent"):
            assert self.task.parent, "Task #{} does not have a parent".format(self.task.id)
            task_id = self.task.parent.id
            data["attrs"]["from_task"] = self.task.id
        else:
            task_id = self.task.id

        resource_meta = data.get("resource_meta")
        if not resource_meta:
            resource_meta = sdk2.Resource[data["type"]].__getstate__()

        resource = controller.Resource.create(
            data["name"], data["file_name"], None, data["type"], task_id, resource_meta, self.task,
            arch=data.get("arch") or ctm.OSFamily.ANY, attrs=data["attrs"],
            system_attributes=data.get("system_attributes")
        )
        return json.dumps(api.Resource.Entry("", "", resource))

    def resources(self, rids):
        return list(map(
            mapping.Resource.to_json,
            filter(None, six.itervalues(mapping.Resource.objects.in_bulk(list(common_itertools.chain(rids)))))
        ))

    def tasks(self, tids):
        return list(map(
            mapping.Task.to_json,
            filter(None, six.itervalues(mapping.Task.objects.in_bulk(list(common_itertools.chain(tids)))))
        ))

    def api_request(self, request):
        from sandbox.yasandbox.controller import dispatch
        task_id, author, method, path, params, headers = request
        # noinspection PyCallingNonCallable,PyArgumentList
        rest_client = dispatch.RestClient(task_id, author=author, return_error=True)()
        # noinspection PyBroadException
        try:
            # noinspection PyProtectedMember
            result = rest_client._request(method, path, params, headers)
        except Exception:
            result = response.HttpExceptionResponse()
        return result.__encode__()


class Dispatcher(ServiceClient):
    SERVICE_APPEARANCE_WAIT = 30

    def __init__(self, host=None, port=None, logger=None, task=None):
        if host is None:
            host = config.Registry().taskbox.dispatcher.server.host
        if port is None:
            port = config.Registry().taskbox.dispatcher.server.port

        self._back_query = BackQueryHandler(task=task)
        super(Dispatcher, self).__init__((host, port), logger)

    @property
    def service_name(self):
        return "dispatcher"

    def ensure_worker(self, tasks_binary_id):
        """ Create new tasks worker. """
        assert isinstance(tasks_binary_id, (int, six.string_types))
        return self("ensure_worker", tasks_binary_id)

    def call(self, tasks_binary_id, request, request_id):
        """ Call taskbox model's method.

        :param int tasks_binary_id: id of resource with tasks binary
        :param sandbox.taskbox.client.protocol.TaskboxRequest request:
        :param str request_id: Request id
        :rtype: `sandbox.taskbox.client.protocol.TaskboxResponse`
        """

        if not isinstance(tasks_binary_id, six.integer_types):
            raise ValueError("Tasks binary id is {!r}, but must be an integer".format(tasks_binary_id))
        tasks_binary_id = int(tasks_binary_id)

        call_args = ("call", tasks_binary_id, request.encode())
        # back compatibility: do not pass `request_id` argument for old binaries
        if request_id is not ctm.NotExists:
            call_args += (request_id,)
        call = self(*call_args, __no_wait=True, __censored=True)
        gen = call.generator
        try:
            rtype, data = gen.next()
            while True:
                rtype, data = gen.send(self._back_query.handle_request(rtype, data))
        except StopIteration:
            data = call.wait()
        return protocol.TaskboxResponse.decode(data, request)


class Worker(ServiceClient):
    # DispatcherServer will watch over worker and retry connection errors
    SERVICE_APPEARANCE_WAIT = 2

    def __init__(self, socket, logger=None):
        super(Worker, self).__init__((socket, None), logger)

    @property
    def service_name(self):
        return "worker"

    def call(self, request_data, request_id):
        """ Call taskbox model's method. """
        call_args = ("call", request_data)
        if request_id is not None:
            call_args += (request_id,)  # TODO: Always pass request_id after all workers update SANDBOX-5358
        return self(*call_args, __no_wait=True, __censored=True)
