import abc
import inspect
import logging
import datetime as dt

import six
from six.moves import cPickle

from sandbox.common import log as common_log
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import format as common_format
from sandbox.common import system as common_system
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics
import sandbox.common.joint.errors as jerrors
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.statistics as ctss

from sandbox import sdk2
from sandbox.sdk2 import internal as sdk2_internal
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context

import sandbox.taskbox.errors as tb_errors
import sandbox.taskbox.statistics as tb_statistics
import sandbox.taskbox.client.service as tb_service
import sandbox.taskbox.client.protocol as tb_protocol

from . import update as model_update


class TaskboxSdkTaskWrapper(object):
    """ Class to wrap sdk2.Task in sdk2.Task.current """

    def __init__(self):
        self.task = None

    @classmethod
    def set_current_task(cls, task):
        if isinstance(sdk2.Task.current, cls):
            sdk2.Task.current.task = task

    def __getattr__(self, item):
        return getattr(self.task, item)


class __BackQueryRestClientMeta(type):
    def __call__(cls, backquery, task_id, author):
        # noinspection PyUnusedLocal
        def _request(self, method, path, params=None, headers=None):
            return backquery.api_request(
                task_id, author, method.__name__, path, params, dict(headers) if headers else headers
            )

        # noinspection PyUnresolvedReferences
        pcls = cls.__base__
        return type(pcls)(pcls.__name__, (pcls,), dict(_request=_request))


@six.add_metaclass(__BackQueryRestClientMeta)
class BackQueryRestClient(common_rest.Client):
    """
    Replaces REST client for back API requests from taskbox.
    """


@six.add_metaclass(abc.ABCMeta)
class BaseTaskBridge(object):
    def __init__(self, model, request_id, logger):
        """
        :type model: `sandbox.yasandbox.database.mapping.Template`
        """
        self.model = model

        self._request_id = request_id
        self._logger = common_log.tracking_logger(logger or logging.getLogger("tb"), request_id)

    @abc.abstractmethod
    def init_model(self):
        pass

    @abc.abstractmethod
    def on_create(self):
        pass

    @abc.abstractmethod
    def on_save(self):
        pass

    @abc.abstractmethod
    def on_enqueue(self):
        pass

    @abc.abstractmethod
    def parameters_meta(self):
        """
        :rtype: mapping.ParametersMeta
        """

    @abc.abstractmethod
    def template_parameters_meta(self):
        """
        :rtype: list(mapping.TaskTemplate.Task.ParameterMeta)
        """

    @abc.abstractmethod
    def template_common_parameters_meta(self):
        """
        :rtype: list(mapping.TaskTemplate.Task.ParameterMeta)
        """

    @abc.abstractmethod
    def template_requirements_meta(self):
        """
        :rtype: list(mapping.TaskTemplate.Task.ParameterMeta)
        """

    @abc.abstractmethod
    def update_parameters(self, data):
        """
        :param data: Tuple of dicts - input and output parameters updates.
        :return: List of validation results, empty if all is ok.
        :rtype: list
        """

    @abc.abstractmethod
    def validate_parameters(self, data):
        """
        :param data: Tuple of dicts - input and output parameters updates.
        :return: List of validation results, empty if all is ok.
        :rtype: list
        """

    @abc.abstractmethod
    def reports(self):
        """
        :rtype: list[mapping.Template.ReportInfo]
        """

    @abc.abstractmethod
    def report(self, report_name):
        """
        :param report_name: Report name: footer or custom name.
        :rtype: object
        """

    @abc.abstractmethod
    def release_template(self):
        """
        :rtype: object
        """

    @abc.abstractmethod
    def hosts_match_score(self, clients):
        """
        :return: List of scores that corresponds to given clients.
        :rtype: list
        """

    @abc.abstractmethod
    def dependent_resources(self):
        pass


class LocalTaskBridge(BaseTaskBridge):
    """ Task bridge to bind mongo model and task instance/class. """

    def __init__(self, model, request_id, setup_agentr, taskbox=None):
        """
        :param mapping.Template model: Task model
        :param str request_id: Request id
        :param setup_agentr: Function that setups agentr
        :param sandbox.taskbox.worker.server.BackQuery taskbox:
        """
        super(LocalTaskBridge, self).__init__(model, request_id, None if taskbox is None else taskbox.job.log)
        self._setup_agentr = setup_agentr
        self._taskbox = taskbox

    @classmethod
    def _notifications(cls, data):
        ns = []
        for item in data:
            statuses = list(common_itertools.chain(ctt.Status.Group.expand(item.get("statuses", ()))))
            for status in statuses:
                if status not in ctt.Status:
                    raise ValueError("Unknown task status {!r}".format(status))
            ns.append(mapping.Task.Notification(
                transport=item.get("transport", ""),
                statuses=statuses,
                recipients=item.get("recipients", ()),
                check_status=item.get("check_status"),
                juggler_tags=item.get("juggler_tags", ())
            ))
        return ns

    @common_patterns.singleton_property
    def cls(self):
        if self.model.type not in sdk2.Task:
            raise common_errors.UnknownTaskType("Unknown or unimplemented SDK2 task type <{}>".format(self.model.type))
        return sdk2.Task[self.model.type]

    def init_model(self):
        cls = self.cls
        self.model.description = self.model.description or cls.Parameters.description.default
        self.model.owner = self.model.owner or cls.Parameters.owner.default
        self.model.max_restarts = self.model.max_restarts or cls.Parameters.max_restarts.default
        self.model.hidden = self.model.hidden or cls.Parameters.hidden.default
        self.model.priority = self.model.priority or cls.Parameters.priority.default
        self.model.kill_timeout = self.model.kill_timeout or cls.Parameters.kill_timeout.default
        self.model.fail_on_any_error = self.model.fail_on_any_error or cls.Parameters.fail_on_any_error.default
        self.model.dump_disk_usage = self.model.dump_disk_usage or cls.Parameters.dump_disk_usage.default
        self.model.tcpdump_args = self.model.tcpdump_args or cls.Parameters.tcpdump_args.default
        self.model.tags = cls.Parameters.tags.default if self.model.tags is None else self.model.tags
        self.model.default_hooks = mapping.Task.DefaultHooks(
            on_create=bool(cls.on_create == sdk2.Task.on_create),
            on_save=bool(cls.on_save == sdk2.Task.on_save),
            on_enqueue=bool(cls.on_enqueue == sdk2.Task.on_enqueue)
        )
        self.model.expires_delta = (
            self.model.expires_delta or
            cls.Parameters.expires.__encode__(cls.Parameters.expires.default)
        )
        self.model.suspend_on_status = self.model.suspend_on_status or cls.Parameters.suspend_on_status.default
        self.model.push_tasks_resource = self.model.push_tasks_resource or cls.Parameters.push_tasks_resource.default
        self.model.score = self.model.score or cls.Parameters.score.default

        self.model.requirements = self.model.requirements or mapping.Task.Requirements()
        self.model.requirements.host = self.model.requirements.host or cls.Requirements.host.default
        self.model.requirements.disk_space = self.model.requirements.disk_space or cls.Requirements.disk_space.default
        self.model.requirements.ram = self.model.requirements.ram or cls.Requirements.ram.default
        self.model.requirements.ramdrive = self.model.requirements.ramdrive or (
            mapping.Task.Requirements.RamDrive(
                type=cls.Requirements.ramdrive.default.type,
                size=cls.Requirements.ramdrive.default.size
            )
            if cls.Requirements.ramdrive.default else
            None
        )
        self.model.requirements.cores = self.model.requirements.cores or cls.Requirements.cores.default
        self.model.requirements.privileged = self.model.requirements.privileged or cls.Requirements.privileged.default
        self.model.requirements.dns = self.model.requirements.dns or cls.Requirements.dns.default
        self.model.requirements.client_tags = (
            self.model.requirements.client_tags or
            cls.Requirements.client_tags.__encode__(cls.Requirements.client_tags.default)
        )
        if cls.Requirements.semaphores.default is not None:
            self.model.requirements.semaphores = (
                self.model.requirements.semaphores or
                mapping.Task.Requirements.Semaphores(**cls.Requirements.semaphores.default.to_dict())
            )

        default_caches = cls.Requirements.Caches.__getstate__()
        self.model.requirements.caches = (
            self.model.requirements.caches or [
                mapping.Task.Requirements.Cache(key=key, value=value)
                for key, value in six.iteritems(default_caches)
            ] if default_caches is not None else None
        )

        self.model.requirements.porto_layers = (
            self.model.requirements.porto_layers or cls.Requirements.porto_layers.default
        )

        if not self.model.requirements.tasks_resource and cls.Requirements.tasks_resource.default:
            self.model.requirements.tasks_resource = mapping.Task.Requirements.TasksResource(
                id=int(cls.Requirements.tasks_resource.default)
            )

        self.model.requirements.container_resource = (
            self.model.requirements.container_resource or cls.Requirements.container_resource.default
        )

        if cls.Parameters.notifications.default is not None:
            self.model.notifications = (
                self.model.notifications or
                self._notifications(cls.Parameters.notifications.__encode__(cls.Parameters.notifications.default))
            )

        if not self.model.context:
            self.model.context = cPickle.dumps(dict(cls.Context), protocol=2)

        self.model.parameters = self.model.parameters or mapping.Task.Parameters()
        if not self.model.parameters.input:
            input_parameters = []
            for p in cls.Parameters:
                if p.dummy or p.__output__:
                    continue
                value = model_update.encode_parameter(p.__encode__(p.default), p.__complex_type__)
                input_parameters.append(mapping.Task.Parameters.Parameter(key=p.name, value=value))
                # TODO: remove after all SDK2 tasks will be fixed [SANDBOX-6188]
                # update container in requirements [SANDBOX-7051]
                if issubclass(p, sdk2.parameters.Container) and not self.model.requirements.container_resource:
                    self.model.requirements.container_resource = value
            self.model.parameters.input = input_parameters
        self.model.reports = model_update.LocalUpdate.task_reports(cls)

    @context.timer_decorator()
    def _update_model_from_task(self, task):
        model_update.LocalUpdate.update_common_fields(task.Parameters.__getstate__(), self.model)
        model_update.LocalUpdate.update_requirements(task, self.model)
        model_update.LocalUpdate.update_context(task, self.model)
        model_update.LocalUpdate.update_parameters(task, self.model)
        model_update.LocalUpdate.update_notifications(task, self.model)
        model_update.LocalUpdate.update_reports(task, self.model)

    def _call_task_method(self, task, method, arguments=(), read_only=False):
        try:
            # Execute user-defined method
            if self._taskbox:
                with common_rest.DispatchedClient as dispatch:
                    dispatch(BackQueryRestClient(self._taskbox, task.id, task.author))
                    return getattr(task, method)(*arguments)
            else:
                return getattr(task, method)(*arguments)
        finally:
            if not read_only:
                # Pull all changes from task object to task model
                self._update_model_from_task(task)

    @context.timer_decorator("LocalTaskBridge.on_create", reset=common_system.inside_the_binary())
    def on_create(self):
        timer = context.current.timer
        with timer["task_from_model"]:
            task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        TaskboxSdkTaskWrapper.set_current_task(task)

        with timer["on_create"]:
            self._call_task_method(task, "on_create")

    @context.timer_decorator("LocalTaskBridge.on_save", reset=common_system.inside_the_binary())
    def on_save(self, task=None):
        timer = context.current.timer
        if task is None:
            with timer["task_from_model"]:
                task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        TaskboxSdkTaskWrapper.set_current_task(task)

        with timer["on_save"]:
            self._call_task_method(task, "on_save")

    @context.timer_decorator("LocalTaskBridge.on_enqueue", reset=common_system.inside_the_binary())
    def on_enqueue(self, task=None):
        timer = context.current.timer
        if task is None:
            with timer["task_from_model"]:
                task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        TaskboxSdkTaskWrapper.set_current_task(task)
        with timer["setup_agentr"]:
            self._setup_agentr(task)

        with timer["validate_params"]:
            param_name = None
            param_value = None
            # TODO: Validation temporary turned off, due to tasks with invalid parameters [SANDBOX-4999]
            try:
                task_params = task.Parameters
                for param in type(task).Parameters:
                    param_name = param.name
                    param_value = getattr(task_params, param.name)
                    if isinstance(param_value, sdk2_internal.parameters.InternalValue):
                        param_value = param_value()
                    param.cast(param_value)
            except (TypeError, ValueError) as ex:
                self._logger.warning(
                    "Invalid value %r for parameter %r found while enqueuing task #%s: %s",
                    param_value, param_name, task.id, ex
                )

        with timer["on_enqueue"]:
            self._call_task_method(task, "on_enqueue")

    def parameters_meta(self):
        return model_update.LocalUpdate.parameters_meta(self.cls.Parameters)

    def template_parameters_meta(self):
        return model_update.LocalUpdate.template_parameters_meta(self.cls.Parameters)

    def template_common_parameters_meta(self):
        cls = self.cls
        return [
            model_update.LocalUpdate.parameter_meta(
                mapping.TaskTemplate.Task.ParameterMeta, getattr(cls.Parameters, attr)
            ) for attr in cls.Parameters.__common_parameters_names__
            if attr not in cls.Parameters.__unused_template_parameters__
        ]

    def template_requirements_meta(self):
        cls = self.cls
        return [
            model_update.LocalUpdate.parameter_meta(
                mapping.TaskTemplate.Task.ParameterMeta, requirement
            ) for requirement in filter(
                lambda x: x.name not in cls.Requirements.__unused_template_requirements__,
                list(cls.Requirements)
            )
        ]

    def tasks_suggest(self, task_type=None):
        task_suggest = []
        tasks = []
        if task_type is not None:
            if task_type in sdk2.Task:
                tasks.append(sdk2.Task[task_type])
            else:
                raise KeyError("Task type {} not found.".format(task_type))
        else:
            tasks = sdk2.Task
        for task in tasks:
            task_suggest.append({
                "type": task.type,
                "owners": task.owners,
                "description": task.description,
                "color": common_format.suggest_color(task.type),
                "client_tags": str(task.Requirements.client_tags.default)
            })
        return task_suggest

    @classmethod
    def _fields_to_drop(cls, parameters_classes, input_dict, output_dict):
        dummy_fields, sub_fields, visible_fields = set(), set(), set()

        for pc in six.viewvalues(parameters_classes):
            if pc.dummy:
                dummy_fields.add(pc.name)
                continue
            output = getattr(pc, "__output__", False)

            # Avoid evaluating `pc.default_value`.
            target_dict = output_dict if output else input_dict
            current_value = target_dict[pc.name] if pc.name in target_dict else pc.default_value

            for v, sf in six.viewitems(getattr(pc, "sub_fields", None) or {}):
                sf = set(sf)
                sub_fields |= sf
                if current_value == v or v in ("false", "true") and str(current_value).lower() == v:
                    visible_fields |= sf
        hidden_fields = dummy_fields | (sub_fields - visible_fields)
        return six.viewkeys(input_dict) & hidden_fields, six.viewkeys(output_dict) & hidden_fields

    @classmethod
    def update_parameters_impl(
        cls, parameters_classes, input_updates, output_updates, input_dict, output_dict, requirements_updates
    ):
        """ Updates parameters regardless task SDK version. """

        parameters_classes = {pc.name: pc for pc in parameters_classes}

        result = tb_protocol.ParametersUpdateResult(errors={})
        input_updated, output_updated = {}, {}
        required, reassigned = set(), set()
        # update container in parameters [SANDBOX-7051]
        container_resource = (requirements_updates or {}).get("container_resource", ctm.NotExists)
        if container_resource is not ctm.NotExists:
            for name, pc in six.iteritems(parameters_classes):
                if pc.ui and pc.ui.type == ctt.ParameterType.CONTAINER:
                    input_updates.setdefault(name, container_resource)
                    break

        for name, value in common_itertools.chain(six.viewitems(input_updates), six.viewitems(output_updates)):
            pc = parameters_classes.get(name)
            if not pc:
                result.errors[name] = "No task field '{}' found".format(name)
                continue

            output = getattr(pc, "__output__", False)
            if (name in input_updates and output) or (name in output_updates and not output):
                # Field is placed to wrong dict, ignore it for back compatibility.
                continue
            if pc.required and not output:
                required.add(pc.name)
            try:
                value = pc.cast(value)
                if value is None and pc.required and not output:
                    result.errors[name] = "Value is required"
                encoder = getattr(pc, "__encode__", None)
                if encoder:
                    value = encoder(value)
            except (ValueError, TypeError) as ex:
                result.errors[name] = "{}: {}".format(ex.__class__.__name__, ex)

            ct = getattr(pc, "__complex_type__", False)
            # New value is saved even in case of cast errors.
            (output_updated if output else input_updated)[name] = model_update.encode_parameter(value, ct)
            if output and name in output_dict:
                current_value = model_update.decode_parameter(output_dict[name], ct)
                # check that output value is not assigned to a different value
                if current_value is not None and value != current_value:
                    reassigned.add(name)

        if reassigned:
            raise tb_errors.OutputParameterReassign(sorted(reassigned))

        input_dict.update(input_updated)
        output_dict.update(output_updated)
        input_to_drop, output_to_drop = cls._fields_to_drop(parameters_classes, input_dict, output_dict)
        for item in input_to_drop:
            del input_dict[item]
        for item in output_to_drop:
            del output_dict[item]

        result.required = sorted(required - {k for k, v in six.viewitems(input_dict) if v is not None})
        result.input_updated = bool(input_updated or input_to_drop)
        result.output_updated = bool(output_updated or output_to_drop)
        return result

    @staticmethod
    def _raw_parameters_dict(parameters_list, classes, output):
        res = {}
        for p in parameters_list:
            pcls = getattr(classes, p.key, None)
            if p.value is not None and pcls is not None and pcls.__output__ == output:
                res[p.key] = p.value
        return res

    @staticmethod
    def _parameters_dict_to_list(parameters_dict):
        p_class = mapping.Task.Parameters.Parameter
        return [p_class(key=name, value=value) for name, value in six.viewitems(parameters_dict)]

    @context.timer_decorator()
    def _update_parameters(self, data, validate_only):

        if self.model.parameters is None:
            self.model.parameters = mapping.Template.Parameters()

        parameters = self.cls.Parameters
        input_dict = self._raw_parameters_dict(self.model.parameters.input, parameters, False)
        output_dict = self._raw_parameters_dict(self.model.parameters.output, parameters, True)

        iu, ou, ru = data
        update_result = self.update_parameters_impl(parameters, iu, ou, input_dict, output_dict, ru)

        if not validate_only:
            if update_result.input_updated:
                self.model.parameters.input = self._parameters_dict_to_list(input_dict)
            if update_result.output_updated:
                self.model.parameters.output = self._parameters_dict_to_list(output_dict)
            model_update.LocalUpdate.update_hints(parameters, self.model)
        return update_result

    def update_hints(self):
        model_update.LocalUpdate.update_hints(self.cls.Parameters, self.model)

    def update_parameters(self, data):
        return self._update_parameters(data, False)

    def validate_parameters(self, data):
        return self._update_parameters(data, True)

    def reports(self):
        return model_update.LocalUpdate.task_reports(self.cls)

    def report(self, report_name):
        task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        try:
            target = getattr(task, self.cls.__reports__[report_name].function_name)
        except (KeyError, AttributeError) as er:
            raise tb_errors.UnknownReportName(str(er))
        return target() if callable(target) else target  # old `footer` function is a property

    def release_template(self):
        task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        return dict(task.release_template)

    def hosts_match_score(self, clients):
        task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        TaskboxSdkTaskWrapper.set_current_task(task)
        return self._call_task_method(task, "hosts_match_score", (clients,), read_only=True)

    @classmethod
    def dependent_resources_from_task(cls, task):
        dependent_resources = set()
        for cls_type in ("Requirements", "Parameters"):
            for param_cls in getattr(task.type, cls_type):
                if not issubclass(param_cls, sdk2.parameters.Resource) or not param_cls.register_dependency:
                    continue
                if issubclass(param_cls, sdk2.parameters.ParentResource):
                    continue

                res = getattr(getattr(task, cls_type), param_cls.name, None)
                if res is None:
                    continue
                # noinspection PyProtectedMember
                if not param_cls._get_contexted_attribute("multiple", "multiple"):
                    res = [res]
                dependent_resources.update(six.moves.map(int, res))
        return list(dependent_resources)

    def dependent_resources(self):
        task = sdk2.Task.from_model(self.model, taskbox=self._taskbox)
        return self.dependent_resources_from_task(task)


class RemoteTaskBridge(BaseTaskBridge):

    def __init__(self, model, request_id):
        super(RemoteTaskBridge, self).__init__(model, request_id, None)
        self._taskbox = tb_service.Dispatcher(logger=self._logger, task=model)

    @staticmethod
    def _get_resolved_field_name(db_field, doc):
        # Build cache for document type
        if not hasattr(doc.__class__, "_field_mapping"):
            # noinspection PyProtectedMember
            doc.__class__._field_mapping = {
                field.db_field: field.name
                for field in doc._fields.values()
            }
        return doc.__class__._field_mapping[db_field]

    def _apply_model_changeset(self, model, changed_fields):
        for field in changed_fields:
            lhs_value = self.model
            rhs_value = model

            path = field.split(".")
            # Descend until last element
            for db_field in path[:-1]:
                resolved_name = self._get_resolved_field_name(db_field, rhs_value)
                lhs_value = getattr(lhs_value, resolved_name)
                rhs_value = getattr(rhs_value, resolved_name)

            # Get last element and copy its value to lhs document
            resolved_name = self._get_resolved_field_name(path[-1], rhs_value)
            # Explicitly reset field to mark its value as "changed"
            setattr(lhs_value, resolved_name, None)
            setattr(lhs_value, resolved_name, getattr(rhs_value, resolved_name))

    def _taskbox_call(
        self, method_name, model=None, read_only=False,
        arg=None, arg_serializer=None, result_serializer=None
    ):
        tr_info = self.model.requirements.tasks_resource
        tasks_binary_id = tr_info.id
        arg_serializer = arg_serializer() if inspect.isclass(arg_serializer) else arg_serializer
        result_serializer = result_serializer() if inspect.isclass(result_serializer) else result_serializer
        request = tb_protocol.TaskboxRequest(
            method_name=method_name,
            model=model,
            task_type=self.model.type if model is None else None,
            read_only=read_only,
            arg=arg,
            arg_serializer=arg_serializer,
            result_serializer=result_serializer,
            age=tr_info.age,
        )
        self._logger.info("Calling %s:%s for #%s", tasks_binary_id, method_name, self.model.id)

        result = tb_statistics.TaskboxRequestResult.SUCCESS
        start_time = dt.datetime.utcnow()

        try:
            resp = self._taskbox.call(
                tasks_binary_id,
                request,
                self._request_id if tr_info.age and tr_info.age >= 4 else ctm.NotExists,
            )

        except Exception as error:
            self._logger.error("Failed to call %s:%s for #%s: %s", tasks_binary_id, method_name, self.model.id, error)
            result = (
                tb_statistics.TaskboxRequestResult.INTERNAL_ERROR
                if isinstance(error, jerrors.ServerError) else
                tb_statistics.TaskboxRequestResult.ERROR
            )
            raise

        finally:
            duration = (dt.datetime.utcnow() - start_time).total_seconds() * 1000
            author, owner = (model.author, model.owner) if model is not None else (None, None)

            common_statistics.Signaler().push(dict(
                type=ctss.SignalType.TASKBOX_CALL,
                date=start_time,
                timestamp=start_time,
                method_name=method_name,
                server=common_config.Registry().this.fqdn,
                duration=duration,
                author=author,
                owner=owner,
                resource_id=tasks_binary_id,
                result=result,
                age=tr_info.age,
            ))

        if not request.read_only and resp.changed_fields and isinstance(resp.model, mapping.Template):
            old_tags = model.tags
            self._apply_model_changeset(resp.model, resp.changed_fields)
            model_update.ServerSideUpdate.postprocess_common_fields(self.model, old_tags=old_tags)

        if resp.exc_info:
            raise sdk2.Wait.decode(resp.exc_info)
        return resp.result

    def init_model(self):
        self._taskbox_call("init_model", model=self.model)

    def on_create(self):
        if not self.model.default_hooks or not self.model.default_hooks.on_create:
            self._taskbox_call("on_create", model=self.model)

    def on_save(self):
        if not self.model.default_hooks or not self.model.default_hooks.on_save:
            self._taskbox_call("on_save", model=self.model)

    def on_enqueue(self):
        if not self.model.default_hooks or not self.model.default_hooks.on_enqueue:
            self._taskbox_call("on_enqueue", model=self.model)

    def parameters_meta(self):
        parameters_meta = self.model.parameters_meta
        if parameters_meta is None:
            parameters_meta = self._taskbox_call(
                "parameters_meta", result_serializer=tb_protocol.ParametersMetaSerializer
            )
            for obj in mapping.ParametersMeta.objects(hash=parameters_meta.calculated_hash):
                if obj.params == parameters_meta.params:
                    parameters_meta = obj
                    break
            if not parameters_meta.id:
                parameters_meta.save()
            else:
                parameters_meta.update(set__accessed=dt.datetime.utcnow())
            self.model.parameters_meta = parameters_meta
            self.model.update(set__parameters_meta=parameters_meta)
        return parameters_meta

    def update_parameters(self, data):
        # TODO: Drop check after switching all binary tasks to newer version SANDBOX-5358
        if self.model.requirements.tasks_resource.age < 5:
            data = data[:2]
        return self._taskbox_call(
            "update_parameters", model=self.model,
            arg=data, result_serializer=tb_protocol.ParametersUpdateSerializer
        )

    def update_hints(self):
        self._taskbox_call("update_hints", model=self.model)

    def validate_parameters(self, data):
        # TODO: Drop check after switching all binary tasks to newer version SANDBOX-5358
        if self.model.requirements.tasks_resource.age < 5:
            data = data[:2]
        return self._taskbox_call(
            "validate_parameters", model=self.model, read_only=True,
            arg=data, result_serializer=tb_protocol.ParametersUpdateSerializer
        )

    def reports(self):
        reports = self.model.reports
        if reports is None:
            # TODO: Drop check after switching all binary tasks to newer version SANDBOX-5358
            if self.model.requirements.tasks_resource.age < 2:
                reports = []
            else:
                reports = self._taskbox_call(
                    "reports", read_only=True, result_serializer=tb_protocol.ReportsInfoSerializer,
                )
            self.model.reports = reports
            self.model.update(set__reports=reports)
        return reports

    def report(self, report_name):
        # TODO: Drop check after switching all binary tasks to newer version SANDBOX-5358
        if self.model.requirements.tasks_resource.age < 2:
            return "Not implemented method. Please rebuild binary task."
        return self._taskbox_call("report", model=self.model, read_only=True, arg=report_name)

    def template_parameters_meta(self):
        return self._taskbox_call(
            "template_parameters_meta", self.model, read_only=True, result_serializer=tb_protocol.TemplateParametersMeta
        )

    def template_common_parameters_meta(self):
        return self._taskbox_call(
            "template_common_parameters_meta", self.model, read_only=True,
            result_serializer=tb_protocol.TemplateParametersMeta
        )

    def template_requirements_meta(self):
        return self._taskbox_call(
            "template_requirements_meta", self.model, read_only=True,
            result_serializer=tb_protocol.TemplateParametersMeta
        )

    def tasks_suggest(self, task_type=None):
        return self._taskbox_call("tasks_suggest", self.model, read_only=True, arg=task_type)

    def release_template(self):
        # TODO: Drop check after switching all binary tasks to newer version SANDBOX-5358
        if self.model.requirements.tasks_resource.age < 2:
            return dict(sdk2.ReleaseTemplate(message="Not implemented method. Please rebuild binary task."))
        return self._taskbox_call("release_template", model=self.model, read_only=True)

    def hosts_match_score(self, clients):
        return self._taskbox_call(
            "hosts_match_score", model=self.model, read_only=True,
            arg=clients, arg_serializer=tb_protocol.ClientListSerializer
        )

    def dependent_resources(self):
        return self._taskbox_call(
            "dependent_resources", model=self.model, read_only=True, result_serializer=tb_protocol.IdentitySerializer,
        )
