import os
import py
import abc
import gevent
import random
import signal
import setproctitle
import logging.handlers

import six

from sandbox.common import log as common_log
from sandbox.common import joint
from sandbox.common import threading as common_threading
from sandbox.common import statistics as common_statistics

import sandbox.common.joint.server as jserver
import sandbox.common.types.misc as ctm

from .. import config


def setup_logger(log_config, log_name, stderr_log=False, formatter=None):
    if stderr_log:
        handler = logging.StreamHandler()
    else:
        py.path.local(log_config.root).ensure_dir()
        log_path = os.path.join(log_config.root, log_name + ".log")
        handler = logging.handlers.WatchedFileHandler(log_path)

    if formatter is None:
        formatter = logging.Formatter("%(asctime)s %(levelname)-7s (%(name)s) %(message)s")
    handler.setFormatter(formatter)

    logging.root.handlers = []  # Somebody could have already initialized logging
    logging.root.addHandler(handler)
    logging.root.setLevel(logging.getLevelName(log_config.level))
    common_statistics.Signaler(
        common_statistics.ClientSignalHandler(token=config.Registry().client.auth.oauth_token),
        component=ctm.Component.TASKBOX,
        update_interval=config.Registry().server.statistics.update_interval,
        config=config.Registry()
    )
    logging.root.addHandler(common_log.ExceptionSignalsSenderHandler())


def setup_worker_logger(log_config, log_name, worker_id, stderr_log=False):
    formatter = logging.Formatter(
        "%(asctime)s %(levelname)-7s {:20} (%(pathname)s:%(lineno)s) %(message)s".format(worker_id)
    )
    setup_logger(log_config, log_name, stderr_log, formatter)


def svn_version():
    import library.python.svn_version as sv
    return sv.svn_version()


def set_signal_handlers(proc_title, jserver):

    def signal_handler(*_):
        setproctitle.setproctitle("{} (finishing...)".format(proc_title))
        jserver.stopping = True

    for sig in (signal.SIGINT, signal.SIGTERM):
        signal.signal(sig, signal_handler)
    signal.signal(signal.SIGUSR2, lambda *_: common_threading.dump_greenlet_threads())


def run_joint_service(server, service_name):
    proc_title = "[sandbox] Taskbox " + service_name
    setproctitle.setproctitle(proc_title)
    set_signal_handlers(proc_title, server)

    server.start()
    try:
        server.loop()
    except KeyboardInterrupt:
        server.stop()


@six.add_metaclass(abc.ABCMeta)
class JointServer(jserver.RPC):
    stopping = False

    def __init__(self, config):
        self._config = config
        self._logger = logging.getLogger(self.service_name)

        ctx = joint.Context(getattr(config.taskbox, self.service_name))
        super(JointServer, self).__init__(ctx)
        self.__server = jserver.Server(ctx)
        self.__server.register_connection_handler(self.get_connection_handler())

    @abc.abstractproperty
    def service_name(self):
        return ""

    def start(self):
        super(JointServer, self).start()
        self.__server.start()
        return self

    def loop(self):
        while not self.stopping:
            gevent.sleep(random.uniform(0.9, 1.1))
        self._logger.info("Main loop finished")
        self.stop()

    def stop(self, graceful=False):
        self._logger.info("Stopping %s", "gracefully" if graceful else "hardly")
        self.__server.stop()
        super(JointServer, self).stop(graceful)
        self._logger.info("Server stopped")
