import json

from sandbox.common import log as common_log
from sandbox.common import itertools as common_itertools
from sandbox.common import projects_handler
import sandbox.common.types.misc as ctm
import sandbox.common.joint.errors as jerrors
import sandbox.common.joint.server as jserver

from sandbox import sdk2

from sandbox.agentr import client as agentr_client

from sandbox.web import response
from sandbox.yasandbox.database import mapping

import sandbox.taskbox.model as tb_model
import sandbox.taskbox.utils as tb_utils
import sandbox.taskbox.client.protocol as tb_protocol


class BackQuery(object):
    def __init__(self, job):
        """
        :param jserver.RPCJob job: Joint RPC job
        """
        self.__job = job

    @property
    def job(self):
        """
        Joint RPC job
        :rtype: jserver.RPCJob
        """
        return self.__job

    def request(self, rtype, data):
        """
        :type rtype: sandbox.common.types.misc.TaskboxBackQuery
        :param data: request payload
        """
        self.__job.state((rtype, data))
        return self.__job.feedback

    def resources(self, rids):
        return list(map(
            mapping.Resource.from_json,
            common_itertools.chain(self.request(ctm.TaskboxBackQuery.RESOURCES, rids) or ())
        ))

    def create_resource(self, data):
        return json.loads(self.request(ctm.TaskboxBackQuery.CREATE_RESOURCES, data))

    def tasks(self, tids):
        return list(map(
            mapping.Task.from_json,
            common_itertools.chain(self.request(ctm.TaskboxBackQuery.TASKS, tids) or ())
        ))

    def api_request(self, task_id, author, method, path, params, headers):
        data = self.request(ctm.TaskboxBackQuery.API_REQUEST, (task_id, author, method, path, params, headers))
        result = response.HttpResponseBase.__decode__(data)
        if isinstance(result, response.HttpErrorResponse):
            result.raise_for_status()
        result.status_code = result.code
        return result


class AgentR(agentr_client.Session):
    """ Replaces some AgentR methods for taskbox-server calls """

    # noinspection PyMissingConstructor
    def __init__(self, backquery, task):
        self.backquery = backquery
        self.task = task

    def __call__(self, method, *args, **kwargs):
        raise NotImplementedError("Method '{}' is not allowed in this context".format(method))

    def resource_register(
        self, path, resource_type, description, arch, attrs,
        share=True, service=False, for_parent=False, depth=None, resource_meta=None, system_attributes=None
    ):
        if for_parent:
            assert self.task.parent, "Task #{} does not have a parent".format(self.task.id)
            task_id = self.task.parent.id
            attrs["from_task"] = self.task.id
        else:
            task_id = self.task.id

        return self.backquery.create_resource(dict(
            type=resource_type,
            name=description,
            file_name=path,
            owner=self.task.owner,
            task_id=task_id,
            arch=arch or ctm.OSFamily.ANY,
            attrs=attrs,
            resource_meta=resource_meta,
            system_attributes=system_attributes
        ))


class Server(tb_utils.JointServer):
    """ Taskbox's worker """

    def __init__(self, config):
        super(Server, self).__init__(config)
        projects_handler.load_project_types()
        sdk2.Task.current = tb_model.TaskboxSdkTaskWrapper()

    @property
    def service_name(self):
        return "worker"

    @jserver.RPC.simple()
    def shutdown(self):
        """ Shutdown server. """
        self.stopping = True

    @jserver.RPC.full()
    def ping(self, ret=True, job=None):
        """ Just returns same value. """
        job.log.info("Pinged with %r value", ret)
        return ret

    @jserver.RPC.dupgenerator()
    def call(self, request_data, request_id, job=None):
        job.log = common_log.tracking_logger(job.log, request_id)

        request = tb_protocol.TaskboxRequest.decode(request_data)
        model = (
            request.model or
            mapping.base.LiteDocument.generate_field_lite_class(mapping.Template)(type=request.task_type)
        )
        taskbox = BackQuery(job) if job else None

        def setup_agentr(task):
            task.agentr = AgentR(taskbox, task)

        bridge = tb_model.LocalTaskBridge(
            model,
            request_id,
            setup_agentr=setup_agentr,
            taskbox=taskbox
        )

        # Base exception for WAIT_* transitions
        from sandbox.sdk2 import Wait

        result, exc_info = None, None
        # Modifications can be observed via `model` property.
        try:
            result = getattr(bridge, request.method_name)(*((request.arg,) if request.arg else ()))

        except Wait as ex:
            # Wait-exceptions are special: they are serialized and returned with response.
            # Caller will reconstruct the exception value and re-raise it to trigger task WAIT_* transition.
            exc_info = ex.encode()

        # noinspection PyProtectedMember
        tb_response = tb_protocol.TaskboxResponse(
            None if request.read_only else bridge.model,
            None if request.read_only else bridge.model._get_changed_fields(),
            request.result_serializer.encode(result),
            exc_info,
        )
        raise jerrors.CallResult(tb_response.encode())
        # it is necessary to make the generator
        # noinspection PyUnreachableCode
        yield
