import abc

from sandbox.common import config as common_config
from sandbox.common import rest as common_rest
from sandbox.common.types import resource as ctr


class BaseHandler:
    """ Base class for handlers """
    def __init__(self, token: str or None, config: common_config.Registry):
        """
        :param token: token for Sandbox session
        :param config: Sandbox config
        """
        self.token = token
        self.config = config
        self.api_client = common_rest.Client(config.client.rest_url, auth=self.token, version=2)

    @abc.abstractmethod
    def download_resource(self, resource_id: int) -> str:
        """ Download resource to some directory and return path to downloaded resource """

    @abc.abstractmethod
    def create_resource(
        self, resource_type: str, arch: str, owner: str, attributes: dict[str, str], path: str, description: str
    ) -> dict:
        """ Create READY resource in Sandbox and return it meta data """

    def _create_api_query(
        self, ids: list[int], resource_type: str, state: ctr.State, owner: str, task_ids: int, any_attr: bool,
        attributes: dict[str, str], offset: int, limit: int, order: list[str]
    ) -> dict:
        """ Format query for Sandbox API GET /resource """
        query = {}
        if ids:
            query["id"] = ids
        if resource_type:
            query["type"] = resource_type
        if state:
            query["state"] = state
        if owner:
            query["owner"] = owner
        if task_ids:
            query["task_id"] = task_ids
        if attributes:
            query["attributes"] = attributes
            query["any_attr"] = any_attr
        if not limit:
            limit = 1
        query["limit"] = limit
        if offset:
            query["offset"] = offset
        if order:
            query["order"] = order
        return query

    def get_resources(
        self, ids: list[int], resource_type: str, state: ctr.State, owner: str, task_ids: int, any_attr: bool,
        attributes: dict[str, str], offset: int, limit: int, order: list[str]
    ) -> list[dict]:
        """ Call GET /resource to Sandbox API to get list of Sandbox resources by query parameters """
        query = self._create_api_query(
            ids, resource_type, state, owner, task_ids, any_attr, attributes, offset, limit, order
        )
        return self.api_client.resource.read(**query)["items"]
