import collections
import logging
import os
import shutil
import threading

import yalibrary.upload.uploader as yauploader

from sandbox.common import config as common_config
from sandbox.common.mds import stream as mds_stream
from sandbox.common import share as common_share
import sandbox.common.types.resource as ctr

from sandbox.tasklet.sidecars.resource_manager.lib import errors
from sandbox.tasklet.sidecars.resource_manager.handlers import base


logger = logging.getLogger("yt_handler")


class YtHandler(base.BaseHandler):
    """ Class for managing Sandbox resources in YT runtime """
    proxy_url = "https://proxy.sandbox.yandex-team.ru"  # Hardcode url for Sandbox proxy
    SKYNET_MIN_RESOURCE_SIZE = 50 << 20  # Minimum resource size for download it via skynet

    class CacheRecord:
        """ Class for description of cache record """
        def __init__(self) -> None:
            self.lock = threading.RLock()
            self.path = None

    def __init__(self, token: str or None, config: common_config.Registry):
        """
        :param token: token for Sandbox session
        :param config: Sandbox config
        """
        super(YtHandler, self).__init__(token, config)
        self.resource_cache = collections.defaultdict(self.CacheRecord)

    def _http_get(self, resource: dict, resource_path: str) -> bool:
        """ Download resource via http from mds. Returns True on success download and False on failed """
        try:
            if resource["md5"] == ctr.EMPTY_FILE_MD5 and resource["size"] == 0 and not resource["multifile"]:
                return False

            return mds_stream.read_from_mds(resource, resource_path)
        except:
            logger.exception("Error on downloading resource #{} via mds.".format(resource["id"]))
            return False

    def _skynet_get(self, resource: dict, resource_path: str) -> bool:
        """ Download resource via skynet. Returns True on success download and False on failed """
        if not resource["skynet_id"]:
            return False
        try:
            common_share.skynet_get(resource["skynet_id"], resource_path, None, resource["size"], fallback_to_bb=True)
            return True
        except:
            logger.exception("Error on downloading resource #{} via skynet.".format(resource["id"]))
            return False

    def _prepare_resource_path(self, path: str) -> None:
        """ Prepare directory for resource: remove all files from it and create top-level directories """
        shutil.rmtree(path, ignore_errors=True)
        os.makedirs(path, exist_ok=True)

    def _download_resource_impl(self, resource_id: int) -> str:
        """ Download Sandbox resource to 'work directory'/'resource_id', and return path to it """
        resources = self.api_client.resource.read(id=[resource_id], limit=1)["items"]
        if not resources:
            raise errors.NotFoundException("Resource #{} not found".format(resource_id))
        resource = resources[0]

        resource_path = os.path.abspath(str(resource_id))
        full_resource_path = os.path.join(resource_path, os.path.basename(resource["file_name"]))
        self._prepare_resource_path(resource_path)
        mds_try = False
        # First try to download small resource via mds
        if resource["size"] < self.SKYNET_MIN_RESOURCE_SIZE:
            if self._http_get(resource, resource_path):
                return full_resource_path
            else:
                mds_try = True
        self._prepare_resource_path(resource_path)
        if self._skynet_get(resource, resource_path) and os.path.exists(full_resource_path):
            return full_resource_path
        self._prepare_resource_path(resource_path)
        if not mds_try and self._http_get(resource, resource_path):
            return full_resource_path
        self._prepare_resource_path(resource_path)
        raise errors.ResourceNotAvailable("Resource #{} not available at MDS and Skynet".format(resource_id))

    def download_resource(self, resource_id: int) -> str:
        """ Download Sandbox resource to 'work directory'/'resource_id', cache it and return path to it """
        cache_record = self.resource_cache[resource_id]

        if cache_record.path is not None:
            return cache_record.path
        with cache_record.lock:
            if cache_record.path is None:
                cache_record.path = self._download_resource_impl(resource_id)

        return cache_record.path

    def create_resource(
        self, resource_type: str, arch: str, owner: str, attributes: dict[str, str], path: str, description: str
    ) -> dict:
        """ Create MDS_UPLOAD task for resource and upload it to Sandbox """
        resource_id = yauploader.do(
            [path],
            resource_type=resource_type,
            resource_arch=arch,
            resource_description=description,
            resource_owner=owner,
            resource_attrs=attributes,
            sandbox_token=self.token
        )
        return self.api_client.resource.read(id=[resource_id], limit=1)["items"][0]
