import aniso8601
import datetime

from sandbox.tasklet.sidecars.resource_manager.proto import resource_pb2
from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2


class ResourceProto:
    """ Methods to convert dict resource objects to proto """
    @staticmethod
    def time(proto_time: resource_pb2.Time, time: dict or None) -> None:
        """ Convert dict resource time to proto resource time """
        if not time:
            return
        now = datetime.datetime.utcnow()
        time_names = ("created", "accessed", "expires", "updated")
        for name in time_names:
            value = aniso8601.parse_datetime(time.get(name)) if time.get(name) else now
            getattr(proto_time, name + "_time").FromDatetime(value)

    @staticmethod
    def mds(proto_mds: resource_pb2.Mds, mds: dict or None) -> None:
        """ Fill proto_mds resource parameters form mds """
        if not mds:
            return
        proto_mds.key = mds.get("key", "") or ""
        proto_mds.namespace = mds.get("namespace", "") or ""
        proto_mds.url = mds.get("url", "") or ""

    @staticmethod
    def resource(proto_resource: resource_pb2.Resource, resource: dict or None) -> None:
        """ Convert dict resource to proto resource """
        if not resource:
            return
        proto_resource.id = resource["id"]
        proto_resource.type = resource["type"]
        proto_resource.arch = resource["arch"]
        proto_resource.state = resource["state"]
        proto_resource.owner = resource["owner"]
        proto_resource.task = resource["task"]["id"]
        proto_resource.description = resource["description"]
        ResourceProto.time(proto_resource.time, resource.get("time"))
        for key, value in resource.get("attributes", {}).items():
            proto_resource.attributes[key] = str(value)
        proto_resource.size = resource.get("size", 0)
        proto_resource.md5 = resource.get("md5", "") or ""
        proto_resource.skynet_id = resource.get("skynet_id", "") or ""
        proto_resource.filename = resource["file_name"]
        ResourceProto.mds(proto_resource.mds, resource.get("mds"))
        proto_resource.multifile = resource.get("multifile", False) or False
        proto_resource.executable = resource.get("executable", False) or False

    @staticmethod
    def resource_array(
        get_resources_response: resource_manager_api_pb2.GetResourcesResponse, resources: list[dict]
    ) -> None:
        """ Convert list of dict resource to repeated proto resources """
        for resource in resources:
            proto_resource = resource_pb2.Resource()
            ResourceProto.resource(proto_resource, resource)
            get_resources_response.resources.append(proto_resource)


class ResourceDict:
    """ Methods to convert proto resource objects to dict """

    @staticmethod
    def new_resource(create_resource_request: resource_manager_api_pb2.CreateResourceRequest) -> dict:
        """ Convert new_resource proto object to dict """
        return {
            "resource_type": create_resource_request.type,
            "arch": create_resource_request.arch,
            "owner": create_resource_request.owner,
            "attributes": dict(create_resource_request.attributes),
            "path": create_resource_request.path,
            "description": create_resource_request.description
        }

    @staticmethod
    def resource_query(get_resources_request: resource_manager_api_pb2.GetResourcesRequest) -> dict:
        """ Convert proto resource query to dict query """
        return {
            "ids": list(get_resources_request.ids),
            "resource_type": get_resources_request.type,
            "state": get_resources_request.state,
            "owner": get_resources_request.owner,
            "task_ids": list(get_resources_request.task_ids),
            "any_attr": get_resources_request.attributes_query.any_attr,
            "attributes": dict(get_resources_request.attributes_query.attributes),
            "offset": get_resources_request.offset,
            "limit": get_resources_request.limit,
            "order": list(get_resources_request.order)
        }
