import os
import shutil
import threading
import json


import sandbox.common.types.resource as ctr

from sandbox.tasklet.sidecars.resource_manager.lib import errors


class Storage:
    STORAGE_PATH = "rm_storage"  # Path to storage
    METADATA_FILE = os.path.join(STORAGE_PATH, "rm_metadata.json")  # Storage metadata file

    """ Class simulates storage of Sandbox resources """
    def __init__(self, path: str = None, resources: dict[int, dict] = None) -> None:
        """ Create storage """
        self.resources = {}
        self.next_id = 1
        self.resource_lock = threading.Lock()
        self.metadata_path = path or os.path.abspath(self.METADATA_FILE)
        if os.path.exists(self.metadata_path):
            with open(self.metadata_path) as f:
                self.resources = {resource["id"]: resource for resource in json.loads(f.read())}
        if resources is not None:
            self.resources.update(resources)
        if self.resources:
            self.next_id = max(self.resources.keys()) + 1

    @staticmethod
    def filter_attributes(resource: dict, query: dict) -> bool:
        """ Check resource attributes by query """
        attributes = query.get("attributes", {})
        if not attributes:
            return True
        any_attr = query.get("any_attr")
        resource_attributes = resource.get("attributes", {})

        for name, value in attributes.items():
            if any_attr and name in resource_attributes and resource_attributes[name] == value:
                return True
            if not any_attr and (name not in resource_attributes or resource_attributes[name] != value):
                return False
        return not any_attr

    @staticmethod
    def filter_ids(resource: dict, query: dict) -> bool:
        """ Check resource id by query """
        ids = query.get("id")
        return not ids or resource["id"] in ids

    @staticmethod
    def filter_task_id(resource: dict, query: dict) -> bool:
        """ Check resource task id by query """
        task_ids = query.get("task_id")
        return not task_ids or resource["task"]["id"] in task_ids

    @staticmethod
    def filter_common_fields(resource: dict, query: dict) -> bool:
        """ Check resource common fields by query """
        ok = True
        ok &= not query.get("owner") or resource["owner"] == query["owner"]
        ok &= not query.get("type") or resource["type"] == query["type"]
        ok &= not query.get("state") or resource["state"] == query["state"]
        return ok

    @staticmethod
    def sort_query(resources: list[dict], order: list[str]) -> None:
        """ Sort list of resources by order fields """
        if not order:
            order = ["-id"]
        order = list(order)
        item = order[0]
        reversed_sort = False
        if item[0] == "-":
            reversed_sort = True
            order[0] = item[1:]
        resources.sort(key=lambda resource: resource[order[0]], reverse=reversed_sort)

    @staticmethod
    def filter_resources(resources: dict[int, dict], query: dict) -> list[dict]:
        """ Simulates API call GET /resource to Sandbox and filter resources by query """
        result = []
        filters = (
            Storage.filter_attributes,
            Storage.filter_ids,
            Storage.filter_task_id,
            Storage.filter_common_fields
        )
        for rid, resource in resources.items():
            if all(f(resource, query) for f in filters):
                result.append(resource)

        Storage.sort_query(result, query.get("order"))
        return result[query.get("offset", 0):query["limit"]]

    def _resource_directory(self, resource_id: int) -> str:
        """ Generate resource directory by resource_id"""
        return os.path.abspath(os.path.join(self.STORAGE_PATH, str(resource_id)))

    def register_resource(
        self, resource_type: str, arch: str, owner: str, attributes: dict[str, str], path: str, description: str
    ) -> dict:
        """ Copy resource file to local storage, generate it metadata and return it """
        with self.resource_lock:
            resource_path = self._resource_directory(self.next_id)
            resource_file_name = os.path.join(resource_path, os.path.basename(path))
            os.makedirs(resource_path)
            if not os.path.exists(path):
                raise OSError("Path '{}' doesn't exists".format(path))
            if os.path.isfile(path):
                shutil.copy2(path, resource_file_name)
            else:
                shutil.copytree(path, resource_file_name)

            resource = {
                "id": self.next_id,
                "type": resource_type,
                "state": ctr.State.READY,
                "arch": arch,
                "owner": owner,
                "attributes": attributes,
                "file_name": os.path.basename(path),
                "description": description,
                "multifile": os.path.isdir(path),
                "task": {"id": self.next_id},
            }
            self.next_id += 1
            self.resources[resource["id"]] = resource
            with open(self.metadata_path, "w") as f:
                f.write(json.dumps(list(self.resources.values())))

            return resource

    def resource_sync(self, resource_id: int) -> str:
        """ Find and return path to resource with id equal to resource_id """
        if resource_id not in self.resources:
            raise errors.NotFoundException("Resource #{} not found".format(resource_id))

        resource = self.resources[resource_id]
        resource_path = self._resource_directory(resource_id)
        resource_file_name = os.path.join(resource_path, os.path.basename(resource["file_name"]))

        if not os.path.exists(resource_file_name):
            raise errors.NotFoundException("Resource #{} not found on filesystem".format(resource_id))

        return resource_file_name
