import os

import grpc

from sandbox.tasklet.sidecars.resource_manager.handlers import sb as sandbox_handler
from sandbox.tasklet.sidecars.resource_manager.handlers import test as test_handler
from sandbox.tasklet.sidecars.resource_manager.handlers import yt as yt_handler
from sandbox.tasklet.sidecars.resource_manager.lib import errors
from sandbox.tasklet.sidecars.resource_manager.lib import model
from sandbox.tasklet.sidecars.resource_manager.lib import types

from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2
from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2_grpc


from sandbox.common import config as common_config


class ResourceManagerAPI(resource_manager_api_pb2_grpc.ResourceManagerAPIServicer):
    def __init__(self, runtime: types.Runtime, iteration: int = None) -> None:
        """
        Service for manage Sandbox resource in different runtimes

        :param runtime: runtime type
        :param iteration: number of Sandbox task restart
        """
        self.token = os.environ.get("SANDBOX_SESSION")
        if runtime == types.Runtime.SANDBOX.value:
            self.resource_handler = sandbox_handler.SandboxHandler(
                self.token, iteration, common_config.Registry()
            )
        elif runtime == types.Runtime.YT.value:
            self.resource_handler = yt_handler.YtHandler(
                self.token, common_config.Registry()
            )
        elif runtime == types.Runtime.TEST.value:
            self.resource_handler = test_handler.TestHandler(
                None, common_config.Registry()
            )
        else:
            raise errors.UnknownRuntimeException(
                "Unknown runtime {}. Available runtimes: {}".format(
                    runtime, ",".join((r.value for r in types.Runtime))
                )
            )

    def DownloadResource(
        self, request: resource_manager_api_pb2.DownloadResourceRequest, context: grpc.ServicerContext
    ) -> resource_manager_api_pb2.DownloadResourceResponse:
        """ Download Sandbox resource and return full path to resource with it file name """
        response = resource_manager_api_pb2.DownloadResourceResponse()
        response.path = self.resource_handler.download_resource(request.id)
        return response

    def CreateResource(
        self, request: resource_manager_api_pb2.CreateResourceRequest, context: grpc.ServicerContext
    ) -> resource_manager_api_pb2.CreateResourceResponse:
        """ Create and complete Sandbox resource and return it meta information"""
        new_resource = model.ResourceDict.new_resource(request)
        response = resource_manager_api_pb2.CreateResourceResponse()
        model.ResourceProto.resource(response.resource, self.resource_handler.create_resource(**new_resource))
        return response

    def GetResources(
        self, request: resource_manager_api_pb2.GetResourcesRequest, context
    ) -> resource_manager_api_pb2.GetResourcesResponse:
        """
        Send GET /resource request to Sandbox API and
        return list of Sandbox resources meta information filtered by query
        """
        resource_query = model.ResourceDict.resource_query(request)
        response = resource_manager_api_pb2.GetResourcesResponse()
        model.ResourceProto.resource_array(response, self.resource_handler.get_resources(**resource_query))
        return response
