#!/usr/bin/env python2

import os
import sys
import shutil
import getpass
import IPython
import logging
import tarfile
import argparse
import tempfile
import compileall

from sandbox import common
import sandbox.common.types.misc as ctm

import sandbox.taskbox.binary as cli


REVISION_FILE_NAME = ".revision"


class CommandHandler(object):

    def __init__(self, args, logger):
        self.args = args
        self.logger = logger

    @common.utils.singleton_property
    def revision(self):
        return common.utils.md5sum(sys.argv[0])

    @common.utils.singleton_property
    def sb(self):
        return cli.Sandbox(cli.Sandbox.BASE_URL, cli.Sandbox.BASE_PROXY_URL, cli.LazyAuth(self.args.token))

    def _embed_revision(self, projects_dir):
        self.logger.debug("Embedding revision %r to %s", self.revision, projects_dir)
        revision_line = "__revision__ = {!r}\n".format(self.revision)
        for dir_name, _, file_names in os.walk(projects_dir):
            with open(os.path.join(dir_name, REVISION_FILE_NAME), "w") as f:
                f.write(self.revision)
            for file_name in file_names:
                if file_name == "__init__.py":
                    file_path = os.path.join(dir_name, file_name)
                    with open(file_path) as f:
                        lines = f.readlines()
                    with open(file_path, "w") as f:
                        written = False
                        for line in lines:
                            if line.startswith("__revision__"):
                                f.write(revision_line)
                                written = True
                            else:
                                f.write(line)
                        if not written:
                            f.write("\n{}".format(revision_line))

    def _make_archive(self, archive_path):
        tmp_dir = tempfile.mkdtemp()
        source_map = cli.dump_sources(tmp_dir)

        proj_dir = "projects"
        sb_proj_dir = os.path.join("sandbox", proj_dir)
        projects_path = os.path.join(tmp_dir, sb_proj_dir)
        modules_inside = list(_ for _ in source_map if _.startswith(sb_proj_dir))
        self.logger.debug("Modules inside:\n%s\n", "\n".join(sorted(modules_inside)))

        resources_inside = cli.dump_resources(tmp_dir, os.path.join("sandbox", "projects"))
        self.logger.debug("Resources inside:\n%s\n", "\n".join(sorted(resources_inside)))

        self._embed_revision(projects_path)
        compileall.compile_dir(projects_path, force=True, quiet=not self.args.verbose)

        with tarfile.open(archive_path, "w:gz") as tar:
            tar.add(projects_path, proj_dir)
            tar.add(os.path.join(projects_path, REVISION_FILE_NAME), REVISION_FILE_NAME)
        if not self.args.debug:
            shutil.rmtree(tmp_dir)

    def validate_args(self):
        if self.args.command.__name__ == self.run_task.__name__:
            task_type = None
            if self.args.tid:
                task_type = self.sb.rest.task[self.args.tid].read()["type"]
            if self.args.template:
                task_type = self.sb.rest.scheduler[self.args.template].read()["task"]["type"]
            if self.args.type:
                task_type = self.args.type
            types = common.projects_handler.load_project_types(raise_exceptions=True)
            if task_type not in types:
                self.logger.error("Task type %r is not defined in tasks archive", task_type)
                sys.exit(1)

    def make_archive(self):
        archive_path = os.path.join(self.args.out, "sandbox-tasks.{}.tar.gz".format(self.revision))
        if os.path.exists(archive_path):
            self.logger.info("Tasks archive %s already exists", archive_path)
        else:
            self._make_archive(archive_path)
            self.logger.info("Tasks archive created and saved to %s", archive_path)
        return archive_path

    def upload_archive(self):
        archive_path = self.make_archive()
        uploader = cli.CommandLineUploader(self.sb, self.logger)
        return uploader.upload(
            archive_path, "SANDBOX_TASKS_ARCHIVE", "Custom Sandbox tasks archive",
            owner=self.args.owner, binary_hash=self.revision, use_skynet=False, use_mds=True,
            extra_attrs=None, force=False,
        )

    def run_task(self):
        resource_id = self.upload_archive()["id"]
        kwargs = {
            "owner": self.args.owner,
            "tasks_archive_resource": resource_id,
        }
        if self.args.tid:
            kwargs["source"] = self.args.tid
        if self.args.template:
            kwargs["scheduler_id"] = self.args.template
        if self.args.type:
            kwargs["type"] = self.args.type
        task = self.sb.rest.task(**kwargs)
        self.logger.info("Task of type %s created: %s", task["type"], self.sb.task_url(task["id"]))
        if self.args.start:
            resp = self.sb.rest.batch.tasks.start.update([task["id"]])
            self.logger.info(resp[0]["message"])

    def ipython(self):
        IPython.start_ipython([])


def _parse_args():
    user = getpass.getuser()
    parser = argparse.ArgumentParser(
        description="Program to manage Sandbox tasks code.", formatter_class=argparse.RawTextHelpFormatter
    )
    subparsers = parser.add_subparsers(title="available commands", help="<description>", metavar="<command>")

    sp_make = subparsers.add_parser(
        "make", help="Make tasks archive.", formatter_class=argparse.RawTextHelpFormatter
    )
    sp_make.set_defaults(command=CommandHandler.make_archive)

    sp_upload = subparsers.add_parser(
        "upload", help="Upload tasks archive.", formatter_class=argparse.RawTextHelpFormatter
    )
    sp_upload.set_defaults(command=CommandHandler.upload_archive)

    sp_run = subparsers.add_parser(
        "run", help="Run task using tasks archive.", formatter_class=argparse.RawTextHelpFormatter
    )
    sp_run.set_defaults(command=CommandHandler.run_task)

    token_default = ctm.Upload.TOKEN_CACHE_FILENAME_TMPL.format(user)
    for p in (sp_upload, sp_run):
        p.add_argument("-o", "--owner", default=user, help="Owner of task (default: {}).".format(user))
        p.add_argument(
            "--token", default=token_default,
            help="Sandbox token or path to file with token (default: {}).".format(token_default),
        )

    sp_run.add_argument("--start", action="store_true", help="Start new task.")

    sp_ipython = subparsers.add_parser("ipython", help="Run ipython using built tasks code.")
    sp_ipython.set_defaults(command=CommandHandler.ipython)

    group = sp_run.add_argument_group("task creation alternatives").add_mutually_exclusive_group(required=True)
    group.add_argument("--tid", metavar="<id>", type=int, help="Task ID to copy.")
    group.add_argument("--template", metavar="<id>", type=int, help="Template (scheduler) ID to use.")
    group.add_argument("--type", help="Task type.")

    for p in (parser, sp_make, sp_upload, sp_run):
        group = p.add_argument_group("common arguments")
        group.add_argument("-v", "--verbose", action="store_true", help="Increase verbosity.")
        group.add_argument("--out", metavar="<path>", default=".", help="Directory to store temporary files.")
        group.add_argument("--debug", action="store_true", help="Do not remove intermediate files.")

    return parser.parse_args()


def main():
    args = _parse_args()
    logger, _ = cli.setup_logging("tasks_run", logging.DEBUG if args.verbose else logging.INFO)
    try:
        handler = CommandHandler(args, logger)
        handler.validate_args()
        args.command(handler)
    except Exception:
        logger.exception("Unhandled exception detected")
        raise


if __name__ == "__main__":
    main()
