import os

import six
import pytest

from sandbox.common import system

from . import utils as tests_utils


pytest_plugins = (
    "sandbox.tests.common.utils",
    "sandbox.tests.common.path",
)


DEFAULT_TEST_MONGO_URI = "mongodb://{}/sandbox_test?replicaSet=sandbox&readPreference=primary".format(
    ",".join("sandbox-dev-dbs{:02d}.search.yandex.net".format(_) for _ in range(1, 4)),
)


def __db_name(prefix, user, hostname):
    # MongoDB only accepts database names shorter than 65 symbols, which can cause trouble
    # with autogenerated names like "sandbox_signal_test_distbuild-man-0055-04_db-runner-key_0-04-1_60422"
    return "_".join((prefix, hostname[:20], user[:10], str(os.getpid())))


@pytest.fixture(scope="session")
def statistics_db_name(request, user):
    hostname = request.config.getoption("--hostname", tests_utils.hostname())
    return __db_name("sandbox_signal_test", user, hostname)


@pytest.fixture(scope="session")
def mongo_uri(request, host, user, statistics_db_name, tests_common_path):
    if system.inside_the_binary():
        return "mongodb://{}:{}/sandbox".format(host, os.environ["RECIPE_MONGO_PORT"])

    hostname = request.config.getoption("--hostname")
    db_suffix = "_{host}_{user}".format(
        host=hostname,
        user=user,
    )

    mongo_uri = request.config.getoption("--mongo-uri", DEFAULT_TEST_MONGO_URI)
    parsed_uri = six.moves.urllib.parse.urlparse(mongo_uri)

    # -> "sandbox_test_{host}_{user}_"
    regular_db_name_prefix = "".join((parsed_uri.path[1:], db_suffix, "_"))
    # -> "sandbox_signal_test_{host}_{user}_"
    signal_db_name_prefix = "{}_".format(statistics_db_name.rsplit("_", 1)[0])

    def remove_old_dbs():
        from sandbox.yasandbox.database import mapping

        uri = six.moves.urllib.parse.urlparse(mongo_uri)._replace(path="test")
        conn = mapping.ensure_connection(uri=uri.geturl()).rw.connection
        for db_name in conn.database_names():
            if any((
                db_name.startswith(prefix)
                for prefix in (regular_db_name_prefix, signal_db_name_prefix)
            )):
                conn.drop_database(db_name)
        mapping.disconnect()

    tests_utils.call_once(request, tests_common_path, remove_old_dbs)

    parsed_uri = parsed_uri._replace(path=__db_name(parsed_uri.path, user, hostname))
    return parsed_uri.geturl()


@pytest.fixture(autouse=True)
def clear_mongo_uri(request, host, user, statistics_db_name, tests_common_path):
    if system.inside_the_binary():
        return "mongodb://{}:{}/sandbox".format(host, os.environ.get("RECIPE_MONGO_PORT"))

    hostname = request.config.getoption("--hostname")

    mongo_uri = request.config.getoption("--mongo-uri", DEFAULT_TEST_MONGO_URI)
    parsed_uri = six.moves.urllib.parse.urlparse(mongo_uri)
    db_name = __db_name(parsed_uri.path, user, hostname)

    def remove_old_dbs():
        from sandbox.yasandbox.database import mapping

        uri = six.moves.urllib.parse.urlparse(mongo_uri)._replace(path="test")
        conn = mapping.ensure_connection(uri=uri.geturl()).rw.connection
        conn.drop_database(db_name.strip("/"))
        mapping.disconnect()

    remove_old_dbs()

    parsed_uri = parsed_uri._replace(path=__db_name(parsed_uri.path, user, hostname))
    return parsed_uri.geturl()
