/* eslint max-statements: [1, 15] */
/* eslint complexity: [1, 13] */

import axios from 'axios';

import actions from '../actions';

const errorHandler = function (dispatch) {
    return function (error) {
        return dispatch({
            type: actions.ALERT_PUSH,
            payload: {
                type: 'error',
                message: error.reason || error.message
            }
        });
    };
};

const fetchSemaphoresList = function (params) {
    params = Object.assign(
        {},
        { limit: 100500, group: '/' },
        params,
        {
            offset: (params && params.page !== 1 && (params.page - 1) * params.limit) ?
                ((params.page - 1) * params.limit) :
                0
        }
    );

    if (params.auto instanceof Array) {
        delete params.auto;
    }

    const isList =
        params.type === 'list' ||
        params.name ||
        params.owner ||
        params.shared ||
        params.auto !== undefined;

    if (isList) {
        delete params.group;
        delete params.page;
        params.limit = params.limit < 100500 ? params.limit : 20;
        params.offset = params.offset || 0;
    } else {
        params.limit = 100500;
        params.offset = 0;
    }

    const type = isList ? 'list' : 'groups';

    delete params.type;

    const request = axios.get('/api/v1.0/semaphore', { params });

    return function (dispatch) {
        dispatch({ type: actions.FETCH_SEMAPHORES });

        dispatch({
            type: actions.SET_CONFIG,
            payload: {
                namespace: 'semaphores',
                key: 'view',
                value: type
            }
        });

        return request
            .then(response => dispatch({
                type: actions.FETCH_SEMAPHORES_SUCCESS,
                payload: {
                    type,
                    group: params.group || '/',
                    data: response.data
                }
            }), errorHandler(dispatch));
    };
};

const fetchSemaphoreById = function (id) {
    const request = axios.get(`/api/v1.0/semaphore/${id}`);

    return function (dispatch) {
        dispatch({
            type: actions.FETCH_SEMAPHORE_BY_ID,
            payload: { id: String(id) }
        });

        return request
            .then(response => dispatch({
                type: actions.FETCH_SEMAPHORE_BY_ID_SUCCESS,
                payload: {
                    id: String(id),
                    data: response.data
                }
            }), errorHandler(dispatch));
    };
};

const fetchSemaphoresAuditById = function (id) {
    const request = axios.get(`/api/v1.0/semaphore/${id}/audit`);

    return function (dispatch) {
        dispatch({
            type: actions.FETCH_SEMAPHORE_AUDIT,
            payload: { id: String(id) }
        });

        return request
            .then(response => dispatch({
                type: actions.FETCH_SEMAPHORE_AUDIT_SUCCESS,
                payload: {
                    id: String(id),
                    data: response.data
                }
            }), errorHandler(dispatch));
    };
};

const fetchSubtleGroup = function (group) {
    const request = axios.get('/api/v1.0/semaphore', { params: { group: group || '/', limit: 100500 }});

    return function (dispatch) {
        dispatch({
            type: actions.FETCH_SEMAPHORES_SUBTLE_GROUP,
            payload: { group }
        });

        return request
            .then(response => dispatch({
                type: actions.FETCH_SEMAPHORES_SUBTLE_GROUP_SUCCESS,
                payload: {
                    group,
                    data: response.data
                }
            }), errorHandler(dispatch));
    };
};

const removeSemaphore = function (group, id) {
    const request = axios.delete(`/api/v1.0/semaphore/${id}`);

    return function (dispatch) {
        dispatch({ type: actions.REMOVE_SEMAPHORE, payload: { id }});

        return request
            .then(() => dispatch({
                type: actions.REMOVE_SEMAPHORE_SUCCESS,
                payload: { id: String(id), group }
            }), errorHandler(dispatch));
    };
};

const pushSemaphore = function (payload) {
    const { id, owner, shared, auto, capacity, event, name } = payload;
    let request;

    if (id) {
        request = axios.put(`/api/v1.0/semaphore/${id}`, { owner, auto, capacity, event, shared });
    } else {
        request = axios.post('/api/v1.0/semaphore/', { owner, shared, capacity, name });
    }

    return function (dispatch) {
        dispatch({ type: actions.PUSH_SEMAPHORE, payload });

        return request
            .then(response => {
                return dispatch({
                    type: actions.PUSH_SEMAPHORE_SUCCESS,
                    payload: id ? payload : response.data
                });
            }, errorHandler(dispatch));
    };
};

module.exports = {
    fetchSemaphoreById,
    fetchSemaphoresAuditById,
    fetchSemaphoresList,
    fetchSubtleGroup,
    removeSemaphore,
    pushSemaphore
};
