const HelperManager = require('./model/service/HelperManager');
const cookie = require('bemmy/src/blocks/component/cookie');
const storage = require('yasap-cache/storage');
const ls = storage.factory({
    gcProbability: 0.01,
    version: window.uiVersion || '0.0.0',
    name: 'name',
    ttl: 3600, // 1 hour
    id: 'url'
});

/**
 *
 * @type     {Marionette.Application}
 * @property {Marionette.Region} content
 */
const app = new Marionette.Application(/** @lends app.prototype */{
    mode: 'normal',
    draftTask: null, // For testing purposes
    draftScheduler: null,
    currentUser: null,
    notifications: [],
    defaultFilter: {},
    externalHelpers: new HelperManager(),
    lastTrace: '',
    autoupdateStatus: Boolean(cookie.get('autoupdatePreference')),
    errXhrObj: null,
    resourceLocator: null,
    performance: {
        internalRoute: '',
        epoch: window.performance.now(),
        header: 0,
        filter: 0,
        content: 0
    },

    /**
     * @returns {HelperManager}
     */
    getHelperManager() {
        return this.externalHelpers;
    },

    getDraftTask() {
        const TaskModel = require('./model/TaskModel');

        if (this.draftTask === null) {
            const saved = ls.get('DRAFT_TASK');

            if (saved) {
                this.draftTask = new TaskModel();
                this.draftTask.set(TaskModel.prototype.parse.call(this.draftTask, saved.data));
            }
        }

        return this.draftTask;
    },

    getDraftScheduler() {
        const SchedulerModel = require('./model/SchedulerModel');

        if (this.draftScheduler === null) {
            const saved = ls.get('DRAFT_SCHEDULER');

            if (saved) {
                this.draftScheduler = new SchedulerModel();
                this.draftScheduler.set(SchedulerModel.prototype.parse.call(this.draftScheduler, saved.data));
            }
        }

        return this.draftScheduler;
    },

    getResourceCreate() {
        const ResourceModel = require('./model/ResourceModel');

        if (this.draftResource === null) {
            const saved = ls.get('DRAFT_RESOURCE');

            if (saved) {
                this.draftResource = new ResourceModel();
                this.draftResource.set(ResourceModel.prototype.parse.call(this.draftResource, saved.data));
            }
        }

        return this.draftResource;
    },

    setDraftTask(task) {
        const TaskModel = require('./model/TaskModel');

        if (task instanceof TaskModel) {
            this.draftTask = task;
            ls.put('DRAFT_TASK', task.toJSON());
        }
    },

    setDraftScheduler(scheduler) {
        const SchedulerModel = require('./model/SchedulerModel');

        if (scheduler instanceof SchedulerModel) {
            this.draftScheduler = scheduler;
            ls.put('DRAFT_SCHEDULER', scheduler.toJSON());
        }
    },

    setDraftResource(resource) {
        const ResourceModel = require('./model/ResourceModel');

        if (resource instanceof ResourceModel) {
            this.draftResource = resource;
            ls.put('DRAFT_RESOURCE', resource.toJSON());
        }
    },

    setLastTrace(trace) {
        this.lastTrace = trace;
    },

    getLastTrace() {
        return this.lastTrace;
    },

    setAutoupdateStatus(flag) {
        this.autoupdateStatus = flag;
    },

    getAutoupdateStatus() {
        return this.autoupdateStatus;
    },

    setErrXhrObj(xhr) {
        this.errXhrObj = xhr;
    },

    getErrXhrObj() {
        return this.errXhrObj;
    },

    dropDraftTask() {
        this.draftTask = null;
        window.localStorage.removeItem('DRAFT_TASK');
    },

    dropDraftScheduler() {
        this.draftScheduler = null;
        window.localStorage.removeItem('DRAFT_SCHEDULER');
    },

    addNotification(notification) {
        this.notifications.push(notification);
    },

    getNotifications() {
        return this.notifications;
    },

    /**
     *
     * @returns {null|UserModel}
     */
    getCurrentUser() {
        return this.currentUser;
    },

    /**
     * @param {UserModel} user
     */
    setCurrentUser(user) {
        const UserModel = require('./model/user');

        if (user instanceof UserModel) {
            this.currentUser = user;
        }
    },

    getDefaultFilter() {
        return this.defaultFilter;
    },

    getCurrentFilter() {
        switch (app.performance.internalRoute) {
            case 'route:home':
                return 'taskList';
            case 'route:resources':
                return 'resourceList';
            case 'route:releases':
                return 'releaseList';
            case 'route:schedulers':
                return 'schedulerList';
            case 'route:clients:task-types':
                return 'taskTypesList';
            case 'route:clients:semaphores':
                return 'semaphoreList';
            case 'route:sandboxNotifications':
                return 'sandboxNotificationsList';
            default:
        }

        return undefined;
    },

    getCurrentCollection() {
        let namespace = 'tasks';

        switch (app.performance.internalRoute) {
            case 'route:home':
                namespace = 'tasks';
                break;
            case 'route:resources':
                namespace = 'resources';
                break;
            case 'route:releases':
                namespace = 'releases';
                break;
            case 'route:schedulers':
                namespace = 'schedulers';
                break;
            default:
        }

        return app.content.currentView.model.attributes[namespace];
    },

    setDefaultFilter(defaultFilter) {
        /* eslint complexity: [1, 10] */
        /* eslint max-statements: [1, 12] */

        if (defaultFilter instanceof require('./model/filter/FilterModel')) {
            this.defaultFilter.taskList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/ResourceFilterModel')) {
            if (defaultFilter.get('attrs')) {
                if (_.isString(defaultFilter.get('attrs'))) {
                    defaultFilter.set('attrs', JSON.parse(defaultFilter.get('attrs')));
                }
            }

            this.defaultFilter.resourceList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/ReleaseFilterModel')) {
            this.defaultFilter.releaseList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/SchedulerFilterModel')) {
            this.defaultFilter.schedulerList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/TaskTypeFilterModel')) {
            this.defaultFilter.taskTypesList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/SemaphoreFilterModel')) {
            this.defaultFilter.semaphoreList = defaultFilter;
        } else if (defaultFilter instanceof require('./model/filter/SandboxNotificationFilterModel')) {
            this.defaultFilter.sandboxNotificationsList = defaultFilter;
        }
    },

    getAppMode() {
        return this.mode;
    },

    setAppMode(mode) {
        this.mode = mode;
    },

    initMetric() {
        if (window.develop) {
            return;
        }

        /** Yandex.Metrika counter */
        (function (d, w, c) {
            (w[c] = w[c] || []).push(() => {
                try {
                    w.yaCounter28049724 = new Ya.Metrika({ id: 28049724,
                        clickmap: true,
                        trackLinks: true,
                        accurateTrackBounce: true });
                } catch (e) { }
            });

            const n = d.getElementsByTagName('script')[0];
            const s = d.createElement('script');
            const f = function () {
                n.parentNode.insertBefore(s, n);
            };

            s.type = 'text/javascript';
            s.async = true;
            s.src = (d.location.protocol === 'https:' ? 'https:' : 'http:') + '//mc.yandex.ru/metrika/watch.js';

            if (w.opera === '[object Opera]') {
                d.addEventListener('DOMContentLoaded', f, false);
            } else {
                f();
            }
        })(document, window, 'yandex_metrika_callbacks');
    },

    trackNotifications(data) {
        data = data || {};

        if (window.Notification) {
            data.status = window.Notification.permission === 'granted' ? 'ENABLED' : 'DISABLED';
        } else {
            data.status = 'UNSUPPORTED';
        }

        this.trackMetricTarget('NOTIFICATION_STATUS', data);
    },

    trackMetricTarget(metricID, metricParams) {
        if (window.yaCounter28049724 && window.yaCounter28049724.reachGoal) {
            window.yaCounter28049724.reachGoal(metricID, metricParams);
        } else {
            const self = this;

            setTimeout(() => {
                self.trackMetricTarget(metricID, metricParams);
            }, 3000);
        }
    },

    getResourceLocator() {
        return this.resourceLocator;
    },

    setResourceLocator(resourceLocator) {
        this.resourceLocator = resourceLocator;
    }
});

app.addInitializer(() => {
    window.addEventListener('storage', evtObj => {
        app.vent.trigger('STORAGE', evtObj);
    });
});

app.addRegions({
    notification: '#notification',
    content: '#content',
    popup: '#popup',
    error: '#error',
    info: '#info'
});

module.exports = app;
