/* eslint max-statements: [1, 12] */
/* eslint complexity: [1, 7] */

const error = require('../error');
const app = require('../app');

/**
 * @class BasicCollection
 * @extends Backbone.Collection
 * @extends Backbone.Events
 *
 * @property {Object} options
 * @property {FilterModel} options.filterParams
 */
const BasicCollection = Backbone.Collection.extend({
    url: '/api/v1.0/',
    model: null,

    isUpdating: false,
    isNeedUpdate: false,

    autoupdateIntervalID: 0,

    options: {
        page: 1,
        total: 1, // No API to get dynamically
        lastSelected: null,
        lastShifted: false,
        filterParams: {},
        pageCapacity: 20,
        selectedItems: []
    },

    initialize(models, options) {
        this.options = _.extend({}, this.options, options);
    },

    onSelectedChange(changedModel) {
        this.options.selectedItems = this.where({
            selected: true
        });

        this.setLastSelected(changedModel);

        app.trigger('change::collection:selected', this.options.selectedItems);
    },

    setSelected(isSelected) {
        this.forEach(task => {
            if (task && task.setSelected) {
                task.setSelected(isSelected);
            }
        });

        this.trigger('change:selected');
    },

    /**
     * @param pageToSet
     * @returns {Boolean|JQueryDeferred}
     */
    setPage(pageToSet) {
        pageToSet = parseInt(pageToSet, 10);

        if (pageToSet && pageToSet > 0) {
            this.options.page = pageToSet;

            return this.update(this.options);
        }

        return false;
    },

    setPageCapacity(pageCapacity) {
        pageCapacity = parseInt(pageCapacity, 10);

        if (pageCapacity && pageCapacity > 0 && pageCapacity !== this.options.pageCapacity) {
            this.options.pageCapacity = pageCapacity;
            this.setPage(1);
        }
    },

    setSortField(sortField, sortType) {
        if (sortType === BasicCollection.SORT_TYPE.CLIENT) {
            this.trigger('update:start');

            const _sortIndex = ((sortField.substring(0, 1) === BasicCollection.SORT_DIR.DESC) ? -1 : 1);
            const _sortKey = sortField.substring(1);

            this.options.order = sortField;
            this.comparator = function (task1, task2) {
                const val1 = task1.get(_sortKey);
                const val2 = task2.get(_sortKey);

                if (val1 < val2) {
                    return _sortIndex * -1;
                }

                if (val1 > val2) {
                    return Number(_sortIndex);
                }

                return 0;
            };

            this.sort();
            this.trigger('update:complete');
        } else if (sortField && typeof sortField === 'string' && sortField !== String(this.options.order)) {
            this.options.order = sortField;
            this.setPage(1);
        }
    },

    setFilterParams(filterParams) {
        if (typeof filterParams === 'object') {
            if (filterParams.get('attr')) {
                if (filterParams.get('attr')) {
                    filterParams.set('attr', _.isString(filterParams.get('attr')) ?
                        JSON.parse(filterParams.get('attr')) :
                        {});
                }
            }

            this.options.filterParams = filterParams;
            this.setPage(1);

            this.trigger('change:filterparams');
        }
    },

    getCurrentItemsKeys() {
        const keys = this.map(item => {
            return item.get('id');
        });

        return {
            id: keys.join(',')
        };
    },

    getFilterParams() {
        return this.options.filterParams;
    },

    getFetchParams(fetchOptions) {
        fetchOptions = _.extend(
            {
                page: this.options.page,
                pageCapacity: this.options.pageCapacity
            },
            fetchOptions
        );

        const result = {
            limit: fetchOptions.pageCapacity,
            offset: (fetchOptions.page - 1) * fetchOptions.pageCapacity
        };

        if (fetchOptions.order) {
            result.order = fetchOptions.order;
        }

        return result;
    },

    getPagingOptions() {
        return {
            page: this.options.page,
            total: this.options.total,
            order: this.options.order,
            pageCapacity: this.options.pageCapacity
        };
    },

    update() {
        app.reqres.request('AUTOUPDATE_SKIP', false);
        throw new Error('SANDBOX: each collection should implement self update logic');
    },

    /**
     * @description will set collection paging options depends on URL values
     * @param {Object} options set of pagination and filter properties
     */
    mapQueryOptions(options) {
        if (options) {
            this.mapPagingQueryOptions(options);
            this.mapFilterQueryOptions(options);
        }
    },

    mapPagingQueryOptions(options) {
        const optsToSet = {};

        if (options.pageCapacity) {
            optsToSet.pageCapacity = parseInt(options.pageCapacity, 10);

            if (optsToSet.pageCapacity <= 0) {
                optsToSet.pageCapacity = this.options.pageCapacity;
            }
        }

        if (options.page) {
            optsToSet.page = parseInt(options.page, 10);

            if (optsToSet.page <= 0) {
                optsToSet.page = 1;
            }
        }

        if (options.order) {
            optsToSet.order = options.order;
        }

        this.options = _.extend(this.options, optsToSet);
    },

    parse(data) {
        this.options.total = parseInt(data.total, 10);
        this.options.pageCapacity = parseInt(data.limit, 10);

        return data.items;
    },

    mapFilterQueryOptions() {
        return null;
    },

    getSelected() {
        return this.options.selectedItems;
    },

    dropLastSelected() {
        this.options.lastSelected = null;
    },

    setLastSelected(lastSelected) {
        if (this.options.lastSelected !== null) {
            if (this.options.lastShifted === true) {
                const from = this.indexOf(lastSelected);
                const to = this.indexOf(this.options.lastSelected);

                if (from !== -1 && to !== -1) {
                    this.slice(from > to ? to : from, from > to ? from : to)
                        .forEach(task => {
                            task.setSelected(true);
                        });
                }
            }
        }

        this.options.lastShifted = false;
        this.options.lastSelected = lastSelected;
    },

    setLastShifted(isShifted) {
        this.options.lastShifted = Boolean(isShifted);
    },

    /**
     * @param {Number} interval - auto update interval in seconds
     */
    activateAutoupdate(interval) {
        const appInterval = app.autoupdateInterval;
        const prevID = app.autoupdateIntervalID;

        app.autoupdateInterval = interval;

        if (appInterval !== interval && prevID) {
            this.deActivateAutoupdate(prevID);
        }

        if (interval === BasicCollection.ONCE) {
            app.reqres.request('AUTOUPDATE_SKIP', true);
            this.update();
        } else if (interval) {
            const self = this;

            app.autoupdateIntervalID = setInterval(() => {
                if (app.autoupdateIntervalID !== prevID) {
                    this.deActivateAutoupdate(prevID);
                }

                if (!app.reqres.request('AUTOUPDATE_SKIP')) {
                    app.reqres.request('AUTOUPDATE_SKIP', true);
                    self.update().done(() => {
                        app.reqres.request('AUTOUPDATE_SKIP', false);
                    });
                }
            }, (interval ? interval * 1000 : BasicCollection.AUTO_UPDATE_INTERVAL));
        }
    },

    deActivateAutoupdate(prevID, isForce) {
        if (isForce) {
            app.reqres.request('AUTOUPDATE_SKIP', false);
        }

        if (prevID) {
            clearInterval(prevID);
            return;
        }

        if (isForce) {
            clearInterval(prevID);
            clearInterval(app.autoupdateIntervalID);
        }
    }

}, {
    batchAction(action, items, options) {
        let result = $.Deferred(); // eslint-disable-line

        options = _.extend({
            url: '',
            idAttr: 'id',
            actions: [],
            successCallback: BasicCollection.applyBatchActionResults
        }, options);

        if (_.values(options.actions).includes(action)) {
            const batchSet = items.map(item => {
                return item.get(options.idAttr);
            });

            result = BasicCollection
                .makeBatchAction(batchSet, action, options.url)
                .done(response => {
                    if (response instanceof Array) {
                        options.successCallback(response, action, items);
                    }
                })
                .fail(error.fromXHR);
        } else {
            result.reject();
        }

        return result;
    },

    makeBatchAction(items, action, url) {
        return $.ajax({
            url,
            data: JSON.stringify(items),
            type: 'PUT',
            dataType: 'json',
            contentType: 'application/json'
        });
    },

    parsePagingProps(query) {
        const parsed = {};

        ['page', 'pageCapacity', 'order'].forEach(key => {
            if (query.hasOwnProperty(key)) { // eslint-disable-line
                parsed[key] = query[key];
            }
        });

        return parsed;
    },

    applyBatchActionResults(processedItems, action, items) {
        const batchResults = {
            SUCCESS: [],
            WARNING: [],
            ERROR: []
        };

        processedItems.forEach(result => {
            const item = _.find(items, item => {
                return (item.get('id') === result.id);
            });

            if (item) {
                batchResults[result.status].push({
                    id: result.id,
                    message: result.message
                });

                item.trigger(BasicCollection.OPERATION_RESULT_EVENT[result.status], {
                    id: result.id,
                    action,
                    message: (result.message ? result.message : '')
                });
            }
        });

        /**
         * Will be triggered on global Backbone object because there is no way
         * to trigger event once on each model's collection.
         */
        if (batchResults.SUCCESS.length || batchResults.WARNING.length || batchResults.ERROR.length) {
            Backbone.trigger(BasicCollection.BATCH_ACTION_RESULT_EVENT, batchResults, action);
        }
    },

    AUTO_UPDATE_INTERVAL: 15000,

    FORCE_AUTO_UPDATE_INTERVAL: 3000
});

BasicCollection.SORT_TYPE = {
    SERVER: 'SERVER',
    CLIENT: 'CLIENT'
};

BasicCollection.OPERATION_RESULT_STATUS = {
    SUCCESS: 'SUCCESS',
    WARNING: 'WARNING',
    ERROR: 'ERROR'
};

BasicCollection.SORT_DIR = {
    ASC: '+',
    DESC: '-'
};

BasicCollection.OPERATION_RESULT_EVENT = {
    ERROR: 'actionError',
    WARNING: 'actionError',
    SUCCESS: 'actionSuccess'
};

/** Will be triggered if on or more batch actions finished with WARN or ERR status */
BasicCollection.BATCH_ACTION_RESULT_EVENT = 'BATCH_ACTION_RESULT';

BasicCollection.ONCE = 'ONCE';

module.exports = BasicCollection;
